import { ApiError, ApiResponse } from "@/@types/api";
import { WishType } from "@/@types/wish";
import { deleteWish } from "@/services/wishlist";
import { useMutation, useQueryClient } from "@tanstack/react-query";
import { AxiosResponse } from "axios";
import Image from "next/image";
import Link from "next/link";
import React from "react";
import CustomToast from "./CustomToast";

export default function WishRow({
  data,
  wishId,
}: {
  data: WishType;
  wishId: string;
}) {
  const { mutate } = useMutation<
    AxiosResponse<ApiResponse<string>>,
    ApiError,
    string
  >({
    mutationFn: deleteWish,
  });
  const queryClient = useQueryClient();
  const removeHandler = () => {
    mutate(wishId || "", {
      onSuccess: (res) => {
        if (res.status === 200) {
          CustomToast({
            type: "success",
            text: "محصول مورد نظر باموفقیت حذف شد",
          });
        }
        queryClient.invalidateQueries({ queryKey: ["wishList"] });
      },
      onError: (error) => {
        const statusCode = error?.response?.status;
        if (statusCode === 400) {
          CustomToast({
            type: "error",
            text: "ورودی معتبر نمی باشد",
          });
        } else if (statusCode === 401) {
          CustomToast({
            type: "error",
            text: "عدم دسترسی به ارسال دیتا",
          });
        } else {
          CustomToast({
            type: "error",
            text: "خطا در انجام عملیات",
            des: "لطفا دوباره امتحان کنید",
          });
        }
      },
    });
  };

  return (
    <tr className=" child:py-[30px] child:px-2 child:text-center child:grow child:min-w-[200px] lg:child:w-[20%]">
      <td>
        <div className="flex items-center gap-2 xl:gap-[18px]">
          <Image
            src={`${process.env.NEXT_PUBLIC_BASE_URL}/${data?.images[0] || ""}`}
            width={50}
            height={50}
            alt="عکس محصول"
            className="w-[50px] h-[50px]"
          />
          <p className="font-peyda text-[#17183B] dark:text-light text-sm xl:text-base">
            {data?.title}
          </p>
        </div>
      </td>
      <td>
        <div className="flex items-center justify-center gap-1.75 text-[#081035] dark:text-light">
          <p className="font-medium text-base xl:text-lg text-center">
            قالب و افزونه
          </p>
        </div>
      </td>
      <td className="text-wrap whitespace-normal text-[#1D1C1A] dark:text-light">
        <p className="text-center text-base">
          {new Date(data?.updatedAt)?.toLocaleDateString("fa-IR")}
        </p>
      </td>

      <td>
        <div className="flex items-center justify-center gap-2.5">
          <span className="inline-block p-1.75 bg-[#3D96D1]/[12%] hover:bg-[#3D96D1]/20 dark:bg-[#3D96D1] dark:hover:bg-[#3D96D1]/80 duration-150 rounded cursor-pointer">
            <Link href={`/products/${data?.title}/${data?.uid}`}>
              <svg className="size-4 text-[#3B93E3] dark:text-light shrink-0">
                <use href="#eye"></use>
              </svg>
            </Link>
          </span>
          <span
            className="inline-block p-1.75 bg-[#EE404C]/[33%] hover:bg-[#EE404C]/40 dark:bg-[#EE404C] dark:hover:bg-[#EE404C]/80  duration-150 rounded cursor-pointer"
            onClick={removeHandler}
          >
            <svg className="size-4 text-[#E04F1A] dark:text-light shrink-0">
              <use href="#trash"></use>
            </svg>
          </span>
        </div>
      </td>
    </tr>
  );
}
