"use client";

import React, { useState } from "react";

import { bankDataType } from "@/@types/Bank";
import LoadingBox from "@/components/modules/LoadingBox";
import { Button } from "@/components/ui/button";
import { BankNameFaHandler, getStatusLabel } from "@/lib/utils";
import { getAdminBanks } from "@/services/bank";
import { useQuery } from "@tanstack/react-query";
import UserBankStatusModal from "./UserBankStatusModal";

type UserBanksTableProps = { userId: string };

export default function UserBanksTable({ userId }: UserBanksTableProps) {
  const [isOpenModal, setIsOpenModal] = useState<boolean>(false);
  const [accountId, setAccountId] = useState<string>("");

  const { data: banks, isLoading } = useQuery({
    queryKey: ["admin-banks"],
    queryFn: () => getAdminBanks(),
    select: (res) => res.data,
  });

  if (isLoading) return <LoadingBox />;
  const [userBanks] = banks?.data.filter(
    (item: { _id: string; banks: bankDataType[] }) => item._id === userId
  );

  return (
    <>
      <table className="w-full mt-7">
        <thead>
          <tr className="child:text-right child:text-[#454545] dark:child:text-light/80 child:text-sm">
            <th className="pb-3.25 !text-center border-b border-dark/[7%] dark:border-light/[6%]">
              نام و نام خانوادگی
            </th>
            <th className="pb-3.25 !text-center border-b border-dark/[7%] dark:border-light/[6%]">
              شماره کارت
            </th>
            <th className="pb-3.25 !text-center border-b border-dark/[7%] dark:border-light/[6%]">
              شماره شبا
            </th>

            <th className="pb-3.25 !text-center border-b border-dark/[7%] dark:border-light/[6%]">
              نام بانک
            </th>
            <th className="pb-3.25 !text-center border-b border-dark/[7%] dark:border-light/[6%]">
              وضعیت
            </th>
            <th className="pb-3.25 !text-center border-b border-dark/[7%] dark:border-light/[6%]">
              عملیات
            </th>
          </tr>
        </thead>
        <tbody className="divide-y divide-dark/[7%] dark:divide-light/[6%]">
          {userBanks?.bank.map((item: bankDataType) => (
            <tr
              key={item._id}
              className=" child:py-[30px] child:px-2 child:text-right child:grow child:w-[175px]"
            >
              <td>
                <p className="text-center text-[#575757] dark:text-light/80 text-sm">
                  {item.accountHolder}
                </p>
              </td>
              <td>
                <p className="text-center text-[#575757] dark:text-light/80 text-sm text-wrap break-all leading-6">
                  {item.cardNumber}
                </p>
              </td>
              <td>
                <p className="text-center text-[#575757] dark:text-light/80 text-sm">
                  {item.shaba}
                </p>
              </td>
              <td>
                <p className="text-center text-[#575757] dark:text-light/80 text-sm">
                  {BankNameFaHandler(item.bankName)}
                </p>
              </td>
              <td>
                <div className="text-center text-[#575757] dark:text-light/80 text-sm">
                  {item.status === "approved" && (
                    <span className="p-1.5 bg-[#00B074]/10 text-xs-mines text-[#00B074] rounded-sm">
                      {getStatusLabel(item.status)}
                    </span>
                  )}
                  {item.status === "pending" && (
                    <span className="p-1.5 bg-[#FF9D00]/10 text-xs-mines text-[#FF9D00] rounded-sm">
                      {getStatusLabel(item.status)}
                    </span>
                  )}
                  {item.status === "rejected" && (
                    <span className="p-1.5 bg-[#FF0000]/10 text-xs-mines text-[#FF0000] rounded-sm">
                      {getStatusLabel(item.status)}
                    </span>
                  )}
                </div>
              </td>
              <td className="!max-w-[80px]">
                {item.status === "pending" || item.status === "rejected" ? (
                  <Button
                    className={`flex-center  w-[70px] p-2.5 bg-[#00B074] hover:bg-[#00B074]/90 duration-150 text-white text-xs mx-auto rounded-md`}
                    onClick={() => {
                      setIsOpenModal(true);
                      setAccountId(item?._id);
                    }}
                  >
                    تأیید
                  </Button>
                ) : (
                  <Button
                    className={`flex-center  w-[70px] p-2.5 bg-[#FF0000] hover:bg-[#FF0000]/90 duration-150 text-white text-xs mx-auto rounded-md`}
                    onClick={() => {
                      setIsOpenModal(true);
                      setAccountId(item?._id);
                    }}
                  >
                    رد
                  </Button>
                )}
              </td>
            </tr>
          ))}
        </tbody>
      </table>
      {isOpenModal && (
        <UserBankStatusModal
          setIsOpenModal={setIsOpenModal}
          devId={userId}
          accountId={accountId}
        />
      )}
    </>
  );
}
