"use client";
import LoadingBox from "@/components/modules/LoadingBox";
import { Button } from "@/components/ui/button";
import { blogSchema } from "@/configs/ValidationSchema";
import { getAdminBlogById, updateBlog } from "@/services/blog";
import { yupResolver } from "@hookform/resolvers/yup";
import { useMutation, useQuery, useQueryClient } from "@tanstack/react-query";
import { useRouter } from "next/navigation";
import React, { useEffect, useState } from "react";
import { Controller, SubmitHandler, useForm } from "react-hook-form";
import * as yup from "yup";
import dynamic from "next/dynamic";
import { AxiosResponse } from "axios";
import { ApiError, ApiResponse } from "@/@types/api";
import CustomToast from "@/components/modules/CustomToast";
import { Checkbox } from "@/components/ui/Checkbox";
import Image from "next/image";
import { UploadType } from "@/@types/Uploade";

type inputs = {
  id: number;
  label: string;
  required: boolean;
  inputType: string;
  textarea?: boolean;
  checkbox?: boolean;
  value:
    | "MetaDescription"
    | "MetaTitle"
    | "dec"
    | "href"
    | "keywords"
    | "title"
    | "alt"
    | "show";
};
type InputsType = yup.InferType<typeof blogSchema>;

const BlogEditor = dynamic(() => import("./BlogEditor"), { ssr: false });

export default function BlogBox({ blogId }: { blogId: string | null }) {
  const [blogContent, setBlogContent] = useState<string>("");
  const [isFileError, setIsFileError] = useState<boolean>(false);
  const [fileError, setFileError] = useState<string | null>(null);
  const [file, setFile] = useState<File[] | null>(null);
  const [isFile, setIsFile] = useState<boolean>(false);
  const [blogUID, setBlogUID] = useState<string>("");
  const [blogImg, setBlogImg] = useState<string>("");
  const [uploadProgress, setUploadProgress] = useState<number>(0);

  const router = useRouter();
  const queryClient = useQueryClient();

  const inputs: inputs[] = [
    {
      id: 1,
      label: "  متادیسکریپشن :",
      required: true,
      inputType: "text",
      value: "MetaDescription",
    },
    {
      id: 2,
      label: "  متا تایتل :",
      required: true,
      inputType: "text",
      value: "MetaTitle",
    },
    {
      id: 3,
      label: "    عنوان :",
      required: true,
      inputType: "text",
      value: "title",
    },
    {
      id: 4,
      label: "لینک مقاله :",
      required: true,
      inputType: "text",
      value: "href",
    },
    {
      id: 5,
      label: " کلمات کلیدی :",
      required: true,
      inputType: "text",
      value: "keywords",
    },
    {
      id: 6,
      label: "توضیحات عکس(alt) :",
      required: true,
      inputType: "text",
      value: "alt",
    },
    {
      id: 7,
      label: "   توضیحات کوتاه :",
      required: true,
      inputType: "text",
      value: "dec",
      textarea: true,
    },
    {
      id: 8,
      label: "محصول فعال",
      required: false,
      inputType: "text",
      checkbox: true,
      value: "show",
    },
  ];

  if (!blogId) {
    router.push("/admin-panel/blogs");
  }

  const { data: blogData, isLoading } = useQuery({
    queryKey: ["blog", blogId || ""],
    queryFn: () => getAdminBlogById(blogId || ""),
    select: (res) => res.data,
  });

  const {
    register,
    handleSubmit,
    reset,
    control,
    formState: { errors },
  } = useForm<InputsType>({
    resolver: yupResolver(blogSchema),
    defaultValues: {
      MetaDescription: "",
      MetaTitle: "",
      dec: "",
      href: "",
      keywords: "",
      title: "",
      alt: "",
      show: false,
    },
  });

  const { mutate, isPending } = useMutation<
    AxiosResponse<ApiResponse<string>>,
    ApiError,
    UploadType
  >({ mutationFn: updateBlog });

  const handleFileChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const fileList = e.target.files;
    if (!fileList) return;

    setIsFileError(false);
    setFileError(null);

    const selectedFiles = Array.from(fileList);
    setFile(selectedFiles);
    e.target.value = "";
  };
  const submitHandler: SubmitHandler<InputsType> = (data) => {
    if (!isFile && (!file || file.length === 0)) {
      setIsFileError(true);
      setFileError("لطفا عکس خود را انتخاب کنید");
      return;
    }
    const formData = new FormData();

    Object.entries(data).forEach(([key, value]) => {
      formData.append(key, String(value));
    });

    const safeFiles = file ?? [];
    safeFiles.forEach((f) => {
      formData.append("img", f);
    });
    formData.append("uid", blogUID);
    formData.append("id", blogId || "");
    formData.append("text", blogContent || "<p><br></p>");

    mutate(
      { data: formData, onProgress: setUploadProgress },
      {
        onSuccess: (res) => {
          if (res.status === 200) {
            CustomToast({
              type: "success",
              text: "مقاله با موفقیت ویرایش شد",
            });
            queryClient.invalidateQueries({ queryKey: ["blog", blogId || ""] });
            setFile(null);
            setUploadProgress(0);
          }
        },
        onError: (error) => {
          const statusCode = error?.response?.status;
          if (statusCode === 400) {
            CustomToast({
              type: "error",
              text: "دیتا وارد شده معتبر نمی باشد",
            });
          } else {
            CustomToast({
              type: "error",
              text: "خطا در انجام عملیات",
              des: "لطفا دوباره امتحان کنید",
            });
          }
        },
      }
    );
  };

  useEffect(() => {
    reset({
      MetaDescription: blogData?.blog?.MetaDescription || "",
      MetaTitle: blogData?.blog?.MetaTitle || "",
      dec: blogData?.blog?.dec || "",
      href: blogData?.blog?.href || "",
      keywords: blogData?.blog?.keywords || "",
      title: blogData?.blog?.title || "",
      alt: blogData?.blog?.alt || "",
      show: blogData?.blog?.show || false,
    });

    setBlogUID(blogData?.blog?.uid || "");
    setBlogContent(blogData?.blog?.text || "");
    setIsFile(blogData?.blog?.img ? true : false);
    setBlogImg(blogData?.blog?.img || "");
  }, [blogData, reset]);

  if (isLoading) return <LoadingBox />;
  return (
    <form onSubmit={handleSubmit(submitHandler)}>
      <div className="grid  grid-cols-1 md:grid-cols-2 xl:grid-cols-3 gap-x-7 xl:gap-x-10 gap-y-7 xl:gap-y-[32px] mt-10">
        {inputs.map((input) => (
          <div
            key={input.id}
            className={`child:text-[#696464] dark:child:text-light ${
              input.textarea ? "col-span-1 md:col-span-2 xl:col-span-3" : ""
            }`}
          >
            {!input.checkbox ? (
              <p className="font text-lg">
                {input.label}{" "}
                {input.required ? (
                  <span className="text-error-400">*</span>
                ) : (
                  ""
                )}
              </p>
            ) : null}
            {input.textarea ? (
              <textarea
                className="w-full  mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 p-4 text-base-mines rounded-sm outline-none"
                placeholder="وارد کنید"
                {...register(input.value)}
                rows={4}
              />
            ) : input.checkbox ? (
              <Controller
                name="show"
                control={control}
                render={({ field }) => (
                  <div className="flex items-center gap-2 h-full mt-5">
                    <Checkbox
                      className="size-6"
                      checked={field.value}
                      onCheckedChange={field.onChange}
                    />
                    <p className="text-lg">مقاله فعال</p>
                  </div>
                )}
              />
            ) : (
              <input
                className="w-full h-[53px] mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 p-4 text-base-mines rounded-sm"
                type={input.inputType}
                placeholder={
                  input?.value === "keywords"
                    ? "کلمات کلیدی رو با , جدا کنید"
                    : "وارد کنید"
                }
                {...register(input.value)}
              />
            )}

            {errors[input.value] && (
              <div className="flex items-center gap-2.5 w-full mt-2.5 child:text-[#FF0000] text-xs">
                <svg className="size-5 shrink-0">
                  <use href="#danger"></use>
                </svg>
                <p className="text-wrap whitespace-normal">
                  {errors[input.value]?.message}
                </p>
              </div>
            )}
          </div>
        ))}
      </div>
      {/* uploader-section */}
      <div className=" my-5 border-y border-light-600/20">
        <div className="my-7">
          <p className="font text-lg child:text-[#696464] dark:child:text-light">
            <span className="font text-lg">عکس آپلود شده وبلاگ:</span>
          </p>
          <div className="relative w-24 h-24 my-5 border rounded overflow-hidden">
            <Image
              src={`${process.env.NEXT_PUBLIC_BASE_URL}${blogImg}`}
              alt={"عکس وبلاگ"}
              width={500}
              height={500}
              className="w-full h-full object-cover"
            />
          </div>
        </div>

        <div className="my-7">
          <p className="font text-lg child:text-[#696464] dark:child:text-light">
            <span className="font text-lg">آپلود عکس :</span>
          </p>
          <label className="flex items-center justify-between w-full sm:w-[180px] py-4 px-6 mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 rounded-md border border-dashed border-[#DDDDDD] dark:border-[#9B9B9B]/50 cursor-pointer">
            <span className="text-[#9B9B9B]">آپلود عکس</span>
            <input
              className="opacity-0 w-0 h-0"
              type="file"
              multiple
              onChange={handleFileChange}
            />
            <svg className="w-6 h-6 text-[#9B9B9B]">
              <use href="#file"></use>
            </svg>
          </label>
          {isFileError && fileError && (
            <div className="flex items-center gap-2.5 w-full mt-2.5 child:text-[#FF0000] text-xs">
              <svg className="size-5 shrink-0">
                <use href="#danger"></use>
              </svg>
              <p className="text-wrap whitespace-normal">{fileError}</p>
            </div>
          )}
        </div>
        <div className="col-span-1 md:col-span-2 xl:col-span-3 flex flex-wrap gap-2 my-2.5">
          {file?.map((f, index) => (
            <div
              key={index}
              className="relative w-24 h-24 border rounded overflow-hidden"
            >
              <Image
                src={URL.createObjectURL(f)}
                alt={f.name}
                width={96}
                height={96}
                className="w-full h-full object-cover"
              />
              <Button
                className="absolute top-1 right-1 bg-[#FF0000] hover:bg-white hover:text-[#FF0000] text-white w-5 h-5 p-0 m-0 flex items-center justify-center rounded-full"
                onClick={() =>
                  setFile((prev) =>
                    prev ? prev.filter((_, i) => i !== index) : null
                  )
                }
              >
                <svg>
                  <use href="#x-mark"></use>
                </svg>
              </Button>
            </div>
          ))}
        </div>
      </div>
      <BlogEditor blogContent={blogContent} setBlogContent={setBlogContent} />
      <div className="flex items-center justify-end gap-1 w-full text-left xl:text-left mt-[23px]">
        <Button
          className="hidden sm:inline-block flex-center w-full xl:w-[165px] h-[55px] px-5 bg-error-400 hover:bg-error-400/90 text-base dark:text-light rounded-10"
          onClick={() => setBlogContent("")}
          type="button"
        >
          پاک کردن همه
        </Button>
        <Button
          className="hidden sm:inline-block flex-center w-full xl:w-[165px] h-[55px] px-5 bg-secondary-default hover:bg-secondary-default/90 text-base dark:text-light rounded-10"
          type="submit"
          disabled={isPending}
        >
          {isPending ? `در حال آپلود ${uploadProgress} %` : "ویرایش"}
        </Button>
      </div>
    </form>
  );
}
