"use client";

import React, { Dispatch, SetStateAction, useEffect, useRef } from "react";
import Quill from "quill";
import "quill/dist/quill.snow.css";
import CustomToast from "@/components/modules/CustomToast";

type BlogEditorType = {
  blogContent: string;
  setBlogContent: Dispatch<SetStateAction<string>>;
};

export default function BlogEditor({
  blogContent,
  setBlogContent,
}: BlogEditorType) {
  const editorRef = useRef<HTMLDivElement>(null);
  const quillRef = useRef<Quill | null>(null);
  const uploadingImagesRef = useRef<Set<string>>(new Set());

  /* -------------------- upload image -------------------- */
  async function uploadImage(file: File): Promise<string> {
    const formData = new FormData();
    formData.append("file", file);

    const res = await fetch(`${process.env.NEXT_PUBLIC_BASE_URL}upload`, {
      method: "POST",
      body: formData,
    });

    if (!res.ok) {
      throw new Error("Upload failed");
    }

    const data = await res.json();
    return data.url;
  }

  /* -------------------- base64 to file -------------------- */
  function base64ToFile(base64: string): File {
    const arr = base64.split(",");
    const mime = arr[0].match(/:(.*?);/)![1];
    const bstr = atob(arr[1]);
    let n = bstr.length;
    const u8arr = new Uint8Array(n);

    while (n--) {
      u8arr[n] = bstr.charCodeAt(n);
    }

    return new File([u8arr], `paste-${Date.now()}.png`, {
      type: mime,
    });
  }

  /* -------------------- toolbar image handler -------------------- */
  function imageHandler() {
    const input = document.createElement("input");
    input.type = "file";
    input.accept = "image/*";
    input.click();

    input.onchange = async () => {
      const file = input.files?.[0];
      if (!file || !quillRef.current) return;

      try {
        CustomToast({
          text: "در حال آپلود",
          des: "لطفا منتظر باشید",
          type: "info",
        });

        const imageUrl = await uploadImage(file);

        CustomToast({
          text: "موفق",
          des: "عکس شما با موفقیت آپلود شد",
          type: "success",
        });

        const range = quillRef.current.getSelection();
        quillRef.current.insertEmbed(range?.index || 0, "image", imageUrl);
      } catch {
        CustomToast({
          text: "ناموفق",
          des: "مشکلی پیش آمده لطفا دوباره امتحان کنید",
          type: "error",
        });
      }
    };
  }

  /* -------------------- init quill -------------------- */
  useEffect(() => {
    if (editorRef.current && !quillRef.current) {
      // eslint-disable-next-line @typescript-eslint/no-explicit-any
      // const Font = Quill.import("formats/font") as any;
      // Font.whitelist = [
      //   "yekanbakh",
      //   "peyda",
      //   "serif",
      //   "monospace",
      //   "sans-serif",
      // ];
      // Quill.register(Font, true);

      // eslint-disable-next-line @typescript-eslint/no-explicit-any
      const Size = Quill.import("formats/size") as any;

      Size.whitelist = [
        "12px",
        "14px",
        "16px",
        "18px",
        "20px",
        "24px",
        "26px",
        "28px",
        "32px",
        "36px",
        "42px",
        "48px",
      ];

      Quill.register(Size, true);

      quillRef.current = new Quill(editorRef.current, {
        theme: "snow",
        modules: {
          toolbar: {
            container: [
              [{ header: [1, 2, 3, 4, 5, 6, false] }],
              [{ size: Size.whitelist }],
              ["bold", "italic", "underline", "strike"],
              [{ color: [] }, { background: [] }],
              [{ script: "sub" }, { script: "super" }],
              [
                { list: "ordered" },
                { list: "bullet" },
                { indent: "-1" },
                { indent: "+1" },
              ],
              [{ direction: "rtl" }, { align: [] }],
              ["blockquote", "code-block"],
              ["link", "image", "video"],
              ["clean"],
            ],
            handlers: {
              image: imageHandler,
            },
          },
        },
      });

      /* -------- handle paste images (Word / Docs) -------- */
      quillRef.current.on("text-change", async () => {
        const quill = quillRef.current!;
        const images = quill.root.querySelectorAll("img");

        Array.from(images).forEach(async (img) => {
          if (
            img.src.startsWith("data:image") &&
            !uploadingImagesRef.current.has(img.src)
          ) {
            uploadingImagesRef.current.add(img.src);

            try {
              CustomToast({
                text: "در حال آپلود تصویر",
                des: "لطفا منتظر بمانید",
                type: "info",
              });

              const file = base64ToFile(img.src);
              const imageUrl = await uploadImage(file);
              img.src = imageUrl;
              CustomToast({
                text: "موفق",
                des: "عکس ها با موفقیت آپلود شد",
                type: "success",
              });
            } catch {
              CustomToast({
                text: "خطا",
                des: "آپلود تصاویر ناموفق بود",
                type: "error",
              });
            }
          }
        });

        setBlogContent(quill.root.innerHTML);
      });
    }
  }, [setBlogContent]);

  /* -------------------- sync external value -------------------- */
  useEffect(() => {
    const quill = quillRef.current;
    if (quill && quill.root.innerHTML !== blogContent) {
      quill.root.innerHTML = blogContent || "";
    }
  }, [blogContent]);

  useEffect(() => {
    if (!quillRef.current || !editorRef.current) return;

    const toolbar = editorRef.current.querySelector(".ql-toolbar");
    if (!toolbar) return;

    // background toolbar
    toolbar.classList.add("dark:bg-gray-800", "dark:border-gray-700");

    // همه دکمه ها و picker ها سفید کن
    const buttons = toolbar.querySelectorAll(
      "button, .ql-picker-label, .ql-picker-item, button svg"
    );
    buttons.forEach((el) => {
      (el as HTMLElement).style.color = "white";
      if (el instanceof SVGElement) el.style.fill = "white";
    });
  }, []); // فقط یکبار بعد از mount اجرا میشه

  return (
    <div
      ref={editorRef}
      style={{
        height: "500px",
        direction: "rtl",
        fontFamily: "var(--font-yekanbakh)",
      }}
      className=" dark:bg-gray-800 dark:child:!text-white"
    />
  );
}
