import { ApiError, ApiResponse } from "@/@types/api";
import { CommentType } from "@/@types/comment";
import CustomToast from "@/components/modules/CustomToast";
import { Button } from "@/components/ui/button";
import { adminAnswerComment } from "@/services/comment";
import { useMutation, useQueryClient } from "@tanstack/react-query";
import { AxiosResponse } from "axios";
import React, { Dispatch, SetStateAction, useState } from "react";
type AnswerCommentModalProps = {
  setIsOpen: Dispatch<SetStateAction<boolean>>;
  commentData: null | CommentType;
  fieldId: string | null;
};
export default function AnswerCommentModal({
  setIsOpen,
  commentData,
  fieldId,
}: AnswerCommentModalProps) {
  const [error, setError] = useState({ errorMsg: "" });
  const [commentText, setCommentText] = useState<string>("");

  const queryClient = useQueryClient();

  const { mutate: answerComment_mutate, isPending: answerComment_isPending } =
    useMutation<
      AxiosResponse<ApiResponse<string>>,
      ApiError,
      {
        fileid: string;
        commentId: string;
        text: string;
      }
    >({
      mutationFn: adminAnswerComment,
    });

  const answerCommentHandler = () => {
    if (!commentText.trim()) {
      setError({ errorMsg: "لطفا پاسخ کامنت خود را بنویسید" });
      return;
    }
    if (commentText.trim().length <= 3) {
      setError({
        errorMsg: "متن کامنت شما باید بیشتر از 3 کاراکتر باشد",
      });
      return;
    }
    setError({ errorMsg: "" });
    answerComment_mutate(
      {
        fileid: fieldId || "",
        commentId: commentData?._id || "",
        // commentId: commentData?._id || "",
        text: commentText,
      },
      {
        onSuccess: (res) => {
          if (res.status === 200) {
            CustomToast({
              type: "success",
              text: "پاسخ کامنت مورد نظر با موفقیت ثبت شد",
            });
          }
          queryClient.invalidateQueries({
            queryKey: ["admin-comments"],
          });
          setIsOpen(false);
        },
        onError: (error) => {
          const statusCode = error?.response?.status;
          if (statusCode === 400) {
            CustomToast({
              type: "error",
              text: "ورودی معتبر نمی باشد",
            });
          } else if (statusCode === 404) {
            CustomToast({
              type: "error",
              text: "کامنت یافت نشد",
            });
          } else {
            CustomToast({
              type: "error",
              text: "خطا در انجام عملیات",
              des: "لطفا دوباره امتحان کنید",
            });
          }
        },
      }
    );
  };

  return (
    <div className="fixed inset-0 flex items-center justify-center mx-auto w-screen h-screen overflow-hidden animate-fadeIn">
      <div className="w-[350px] md:w-[400px]w-[350px] md:w-[600px] p-3 bg-light dark:bg-dark-400 rounded-10 z-50">
        <div>
          <p className="font-bold text-xl text-[#001A46] dark:text-light">
            پاسخ کامنت
          </p>
          <div className="flex items-center mt-4">
            <span className="w-[25%] h-0.5 bg-secondary-default dark:bg-primary-200"></span>
            <span className="w-[75%] h-0.5 bg-dark/[7%] dark:bg-[#4A4A4A]"></span>
          </div>
        </div>
        <p className="px-1 pt-2.5 text-sm text-light-600">
          از طرف {`${commentData?.userid?.fname} ${commentData?.userid?.lname}`}
        </p>
        <p className="py-5 px-8 m-2.5 border border-dark/[10%] dark:border-light/[6%] rounded-9">
          {commentData?.text}
        </p>

        <textarea
          className="w-full mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 p-4 text-base-mines rounded-sm outline-none"
          rows={5}
          placeholder="پاسخ خود را وارد کنید ..."
          value={commentText}
          onChange={(e) => setCommentText(e.target.value)}
        />
        {!!error?.errorMsg && (
          <div className="flex items-center gap-2.5 w-full my-2.5 child:text-[#FF0000] text-xs">
            <svg className="size-5 shrink-0">
              <use href="#danger"></use>
            </svg>
            <p className="text-wrap whitespace-normal">{error?.errorMsg}</p>
          </div>
        )}
        <div className="flex-center w-full mt-1 gap-2">
          <Button
            className="inline-block flex-center w-20 h-10 px-5 bg-error-400 hover:bg-error/90 text-base dark:text-light rounded"
            onClick={() => setIsOpen(false)}
          >
            لغو
          </Button>

          <Button
            className="inline-block flex-center w-20 h-10 px-5 bg-secondary-default hover:bg-secondary-default/90 text-base dark:text-light rounded"
            type="submit"
            disabled={answerComment_isPending}
            onClick={answerCommentHandler}
          >
            ثبت
          </Button>
        </div>
      </div>
      <div
        className="absolute w-full h-full z-40"
        onClick={() => {
          setIsOpen(false);
        }}
      ></div>
    </div>
  );
}
