import { ApiError, ApiResponse } from "@/@types/api";
import CustomToast from "@/components/modules/CustomToast";
import { Button } from "@/components/ui/button";
import { deleteContactUsMsg } from "@/services/contact";
import { useMutation, useQueryClient } from "@tanstack/react-query";
import { AxiosResponse } from "axios";
import React, { Dispatch, SetStateAction } from "react";

type ContactUsMsgDeleteModalProps = {
  setIsOpenDeleteModal: Dispatch<SetStateAction<boolean>>;
  msgId: string;
};

export default function ContactUsMsgDeleteModal({
  setIsOpenDeleteModal,
  msgId,
}: ContactUsMsgDeleteModalProps) {
  const queryClient = useQueryClient();
  const { mutate, isPending } = useMutation<
    AxiosResponse<ApiResponse<string>>,
    ApiError,
    string
  >({ mutationFn: deleteContactUsMsg });

  const clickHandler = () => {
    mutate(msgId, {
      onSuccess: () => {
        CustomToast({
          type: "success",
          text: "پیام کاربر با موفقیت حذف شد",
        });
        queryClient.invalidateQueries({ queryKey: ["admin-contact-us"] });
        setIsOpenDeleteModal(false);
      },
      onError: (error) => {
        const statusCode = error?.response?.status;
        if (statusCode === 404) {
          CustomToast({
            type: "error",
            text: "پیام کاربر پیدا نشد",
          });
        } else if (statusCode === 400) {
          CustomToast({
            type: "error",
            text: "شناسه نامعتبر",
          });
        } else {
          CustomToast({
            type: "error",
            text: "خطا در انجام عملیات",
            des: "لطفا دوباره امتحان کنید",
          });
        }
      },
    });
  };
  return (
    <div className="fixed inset-0 flex items-center justify-center mx-auto w-screen h-screen overflow-hidden animate-fadeIn z-50">
      <div className="w-[350px] md:w-[400px] p-3 bg-light dark:bg-dark-400 rounded-10 z-50">
        <div>
          <p className="font-bold text-base text-[#001A46] dark:text-light">
            حذف پیام کاربر
          </p>
          <div className="flex items-center mt-4">
            <span className="w-[20%] h-0.5 bg-secondary-default dark:bg-primary-200"></span>
            <span className="w-[80%] h-0.5 bg-dark/[7%] dark:bg-[#4A4A4A]"></span>
          </div>
        </div>
        <p className="my-3.5 font-peyda font-bold text-center">
          آیا از حذف پیام کاربر مطمئن هستید؟
        </p>
        <div className="flex-center gap-2.5 w-full py-2">
          <Button
            className="inline-block flex-center h-10 px-5 bg-secondary-default hover:bg-secondary-default/90 text-base text-light hover:text-light rounded"
            variant="outline"
            onClick={() => {
              setIsOpenDeleteModal(false);
            }}
          >
            لغو
          </Button>
          <Button
            className="inline-block flex-center h-10 px-5 bg-red-600 hover:bg-red-500 text-base text-light rounded"
            onClick={clickHandler}
            disabled={isPending}
          >
            حذف
          </Button>
        </div>
      </div>

      <div
        className="absolute w-full h-full z-40"
        onClick={() => {
          setIsOpenDeleteModal(false);
        }}
      ></div>
    </div>
  );
}
