"use client";
import { ApiError, ApiResponse } from "@/@types/api";
import { CategoryType } from "@/@types/categorey";
import CustomToast from "@/components/modules/CustomToast";
import { Button } from "@/components/ui/button";
import { Checkbox } from "@/components/ui/Checkbox";
import { FileSchema } from "@/configs/ValidationSchema";
import { getAdminCategory, getAdminSubCategory } from "@/services/category";
import { sendAdminFile } from "@/services/file";
import { yupResolver } from "@hookform/resolvers/yup";
import { useMutation, useQuery, useQueryClient } from "@tanstack/react-query";
import { AxiosResponse } from "axios";
import React, { useState } from "react";
import {
  Controller,
  SubmitHandler,
  useFieldArray,
  useForm,
} from "react-hook-form";
import * as yup from "yup";
import dynamic from "next/dynamic";
import Image from "next/image";
import { UploadType } from "@/@types/Uploade";
import { api } from "@/configs/api";

const BlogEditor = dynamic(() => import("../blogs/blog/BlogEditor"), {
  ssr: false,
});

type InputsType = yup.InferType<typeof FileSchema>;
type inputs = {
  id: number;
  label: string;
  required: boolean;
  inputType: string;
  textarea?: boolean;
  checkbox?: boolean;
  value:
    | "title"
    | "subdecs"
    | "decs"
    | "price"
    | "categoryid"
    | "subcategoryid"
    | "previewlink"
    | "customOptions"
    | "features"
    | "isaccept";
};

export default function FileForm() {
  const {
    register,
    handleSubmit,
    formState: { errors },
    control,
    watch,
    reset,
  } = useForm<InputsType>({
    resolver: yupResolver(FileSchema),
    defaultValues: {
      title: "",
      subdecs: "",
      decs: "",
      price: "",
      categoryid: "",
      subcategoryid: "",
      previewlink: "",
      isaccept: false,
      features: [{ name: "", detail: "" }],
      customOptions: [{ title: "", value: "" }],
    },
  });
  const {
    fields: featureFields,
    append: addFeature,
    remove: removeFeature,
  } = useFieldArray({
    control,
    name: "features",
  });

  const {
    fields: optionFields,
    append: addOption,
    remove: removeOption,
  } = useFieldArray({
    control,
    name: "customOptions",
  });
  const categoryId = watch("categoryid");

  const { data } = useQuery({
    queryKey: ["admin-category"],
    queryFn: () => getAdminCategory(),
    select: (res) => res.data,
  });
  const { data: subCategory_data } = useQuery({
    queryKey: ["admin-subcategory", categoryId],
    queryFn: () => getAdminSubCategory(String(categoryId)),
    select: (res) => res?.data,
    enabled: !!categoryId,
  });

  const inputs: inputs[] = [
    {
      id: 1,
      label: " عنوان :",
      required: false,
      inputType: "text",
      value: "title",
    },
    {
      id: 4,
      label: "قیمت :",
      required: false,
      inputType: "text",
      value: "price",
    },
    {
      id: 7,
      label: "لینک پیش نمایش :",
      required: false,
      inputType: "text",
      value: "previewlink",
    },
    {
      id: 11,
      label: "پیش نمایش فعال",
      required: false,
      inputType: "text",
      checkbox: true,
      value: "isaccept",
    },
    // {
    //   id: 12,
    //   label: "  توضیحات بیشتر :",
    //   required: false,
    //   inputType: "text",
    //   value: "subdecs",
    //   textarea: true,
    // },
    {
      id: 13,
      label: " توضیحات کوتاه :",
      required: false,
      inputType: "text",
      value: "subdecs",
      textarea: true,
    },
  ];

  const [fileError, setFileError] = useState<string | null>(null);
  const [file, setFile] = useState<File[] | null>(null);
  const [isFileError, setIsFileError] = useState<boolean>(false);

  const [coverError, setCoverError] = useState<string | null>(null);
  const [cover, setCover] = useState<File[] | null>(null);
  const [isCoverError, setIsCoverError] = useState<boolean>(false);

  const [productFileError, setProductFileError] = useState<string | null>(null);
  const [productFile, setProductFile] = useState<string | null>(null);
  const [isProductFileError, setIsProductFileError] = useState<boolean>(false);
  const [productFileName, setProductFilename] = useState<string | null>(null);

  const [uploading, setUploading] = useState(false);
  const [uploadProductFileProgress, setUploadProductFileProgress] =
    useState<number>(0);

  const [uploadProgress, setUploadProgress] = useState<number>(0);

  const productFileInputRef = React.useRef<HTMLInputElement | null>(null);

  const queryClient = useQueryClient();

  const { mutate, isPending } = useMutation<
    AxiosResponse<ApiResponse<string>>,
    ApiError,
    UploadType
  >({ mutationFn: sendAdminFile });

  const MAX_FILES = 10;
  const MAX_FILE_SIZE_MB = 1;

  const handleFileChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const files = e.target.files;
    if (!files) return;

    setIsFileError(false);
    setFileError(null);

    const selectedFiles = Array.from(files);

    // محدودیت تعداد
    if (selectedFiles.length + (file?.length || 0) > MAX_FILES) {
      setIsFileError(true);
      setFileError(`حداکثر ${MAX_FILES} فایل می‌توانید انتخاب کنید`);
      return;
    }

    // محدودیت حجم
    const invalidFile = selectedFiles.find(
      (f) => f.size / 1024 / 1024 > MAX_FILE_SIZE_MB
    );
    if (invalidFile) {
      setIsFileError(true);
      setFileError(
        `حجم فایل‌ها نباید بیشتر از ${MAX_FILE_SIZE_MB} مگابایت باشد`
      );
      return;
    }

    // اضافه کردن فایل‌ها به استیت
    setFile((prev) => (prev ? [...prev, ...selectedFiles] : selectedFiles));
    e.target.value = "";
  };

  const MAX_COVER = 1;

  const handleCoverChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const files = e.target.files;
    if (!files) return;

    setIsCoverError(false);
    setCoverError(null);

    const selectedFiles = Array.from(files);

    // محدودیت تعداد
    if (selectedFiles.length + (cover?.length || 0) > MAX_COVER) {
      setIsCoverError(true);
      setCoverError(`حداکثر ${MAX_COVER} فایل می‌توانید انتخاب کنید`);
      return;
    }

    // محدودیت حجم
    const invalidFile = selectedFiles.find(
      (f) => f.size / 1024 / 1024 > MAX_FILE_SIZE_MB
    );
    if (invalidFile) {
      setIsCoverError(true);
      setCoverError(
        `حجم عکس ها نباید بیشتر از ${MAX_FILE_SIZE_MB} مگابایت باشد`
      );
      return;
    }

    // اضافه کردن عکس ها به استیت
    setCover((prev) => (prev ? [...prev, ...selectedFiles] : selectedFiles));
    e.target.value = "";
  };

  const MAX_PRODUCT_FILE_SIZE_MB = 500;
  const ALLOWED_EXTENSIONS = ["zip", "pdf", "rar", "svg", "png", "jpg", "jpeg"];

  const uploadFile = async (file: File): Promise<string> => {
    const formData = new FormData();
    formData.append("file", file);

    const res = await api.post(
      `${process.env.NEXT_PUBLIC_BASE_URL}upload`,
      formData,
      {
        onUploadProgress: (progressEvent) => {
          if (!progressEvent.total) return;

          const percent = Math.round(
            (progressEvent.loaded * 100) / progressEvent.total
          );
          setUploadProductFileProgress(percent);
        },
      }
    );

    return res.data.url;
  };

  const handleProductFileChange = async (
    e: React.ChangeEvent<HTMLInputElement>
  ) => {
    const files = e.target.files;
    if (!files) return;

    setIsProductFileError(false);
    setProductFileError(null);

    const selectedFile = Array.from(files);

    // محدودیت تعداد
    if (productFile) {
      setIsProductFileError(true);
      setProductFileError("ابتدا فایل قبلی را حذف کنید");
      return;
    }
    const fileExt = selectedFile[0].name.split(".").pop()?.toLowerCase();
    if (!fileExt || !ALLOWED_EXTENSIONS.includes(fileExt)) {
      setIsProductFileError(true);
      setProductFileError(
        "فرمت فایل مجاز نیست. لطفاً یکی از فرمت‌های Zip, PDF, RAR, SVG, PNG, JPG انتخاب کنید"
      );
      e.target.value = "";
      return;
    }

    // محدودیت حجم
    const invalidFile = selectedFile.find(
      (f) => f.size / 1024 / 1024 > MAX_PRODUCT_FILE_SIZE_MB
    );
    if (invalidFile) {
      setIsProductFileError(true);
      setProductFileError(
        `حجم فایل نباید بیشتر از ${MAX_PRODUCT_FILE_SIZE_MB} مگابایت باشد`
      );
      return;
    }

    // اضافه کردن فایل  به استیت
    try {
      setUploading(true);
      CustomToast({
        type: "info",
        text: "در حال آپلود فایل...",
      });
      const fileUrl = await uploadFile(selectedFile[0]);

      setProductFile(fileUrl ?? null);
      setProductFilename(selectedFile[0]?.name);

      CustomToast({
        type: "success",
        text: "فایل با موفقیت آپلود شد",
      });
    } catch {
      CustomToast({
        type: "error",
        text: "خطا در آپلود فایل",
      });
      setUploadProductFileProgress(0);
    } finally {
      setUploading(false);
      e.target.value = "";
    }
  };

  const submitHandler: SubmitHandler<InputsType> = (data) => {
    if (!file || file.length === 0) {
      setIsFileError(true);
      setFileError("لطفا فایل خود را انتخاب کنید");
      return;
    }

    if (!cover || cover.length === 0) {
      setIsCoverError(true);
      setCoverError("لطفا عکس کاور خود را انتخاب کنید");
      return;
    }

    const formData = new FormData();

    const cleanedFeatures =
      data.features?.filter(
        (item) => item.name?.trim() || item.detail?.trim()
      ) || [];

    const cleanedOptions =
      data.customOptions?.filter(
        (item) => item.title?.trim() || item.value?.trim()
      ) || [];

    Object.entries(data).forEach(([key, value]) => {
      if (key === "features") {
        formData.append("features", JSON.stringify(cleanedFeatures));
      } else if (key === "customOptions") {
        formData.append("customOptions", JSON.stringify(cleanedOptions));
      } else {
        formData.append(key, String(value));
      }
    });

    file.forEach((f) => {
      formData.append("file", f);
    });

    cover.forEach((c) => {
      formData.append("cover", c);
    });

    mutate(
      { data: formData, onProgress: setUploadProgress },
      {
        onSuccess: (res) => {
          if (res.status === 200) {
            CustomToast({
              type: "success",
              text: "محصول جدید با موفقیت ایجاد شد",
            });
            setFile(null);
            queryClient.invalidateQueries({ queryKey: ["admin-files"] });
            reset(
              {
                title: "",
                subdecs: "",
                decs: "",
                price: "",
                categoryid: "",
                subcategoryid: "",
                previewlink: "",
                isaccept: false,
                features: [{ name: "", detail: "" }],
                customOptions: [{ title: "", value: "" }],
              },
              { keepDefaultValues: false }
            );
            setFile(null);
            setCover(null);
            setUploadProgress(0);
          }
        },
        onError: (error) => {
          const statusCode = error?.response?.status;
          if (statusCode === 400) {
            CustomToast({
              type: "error",
              text: "دیتا وارد شده معتبر نمی باشد",
            });
          } else {
            CustomToast({
              type: "error",
              text: "خطا در انجام عملیات",
              des: "لطفا دوباره امتحان کنید",
            });
          }
          setUploadProgress(0);
        },
      }
    );
  };

  const removeUploadFileHandler = () => {
    setProductFile(null);
    setUploadProductFileProgress(0);
    setIsProductFileError(false);
    setProductFileError(null);
    if (productFileInputRef.current) {
      productFileInputRef.current.value = "";
    }
  };

  return (
    <form onSubmit={handleSubmit(submitHandler)}>
      <div className="grid  grid-cols-1 md:grid-cols-2 xl:grid-cols-3 gap-x-7 xl:gap-x-10 gap-y-7 xl:gap-y-[32px] mt-10">
        {inputs.map((input) => (
          <div
            key={input.id}
            className={`child:text-[#696464] dark:child:text-light ${
              input.textarea ? "col-span-1 md:col-span-2 xl:col-span-3" : ""
            }`}
          >
            {!input.checkbox ? (
              <p className="font text-lg">
                {input.label}{" "}
                {input.required ? (
                  <span className="text-error-400">*</span>
                ) : (
                  ""
                )}
              </p>
            ) : null}
            {input.textarea ? (
              <textarea
                className="w-full  mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 p-4 text-base-mines rounded-sm outline-none"
                placeholder="وارد کنید"
                {...register(input.value)}
                rows={4}
              />
            ) : input.checkbox ? (
              <Controller
                name="isaccept"
                control={control}
                render={({ field }) => (
                  <div className="flex items-center gap-2 h-full mt-5">
                    <Checkbox
                      className="size-6"
                      checked={field.value}
                      onCheckedChange={field.onChange}
                    />
                    <p className="text-lg">پیش نمایش فعال</p>
                  </div>
                )}
              />
            ) : (
              <input
                className="w-full h-[53px] mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 p-4 text-base-mines rounded-sm"
                type={input.inputType}
                placeholder="وارد کنید"
                {...register(input.value)}
              />
            )}

            {errors[input.value] && (
              <div className="flex items-center gap-2.5 w-full mt-2.5 child:text-[#FF0000] text-xs">
                <svg className="size-5 shrink-0">
                  <use href="#danger"></use>
                </svg>
                <p className="text-wrap whitespace-normal">
                  {errors[input.value]?.message}
                </p>
              </div>
            )}
          </div>
        ))}
        <div className="child:text-[#696464] dark:child:text-light">
          <p className="font text-lg">
            <span className="font text-lg">دسته بندی محصول :</span>
          </p>

          <select
            className="w-full h-[53px] mt-2.25 bg-[#EFEFEF] dark:bg-[#535353] p-4 pl-10 text-base-mines rounded-sm outline-none"
            {...register("categoryid")}
          >
            <option value="-1" defaultChecked>
              انتخاب کنید
            </option>
            {data?.categorys?.map((category: CategoryType) => (
              <option key={category._id} value={category._id}>
                {category.title}
              </option>
            ))}
          </select>
          {errors.categoryid && (
            <div className="flex items-center gap-2.5 w-full mt-2.5 child:text-[#FF0000] text-xs">
              <svg className="size-5 shrink-0">
                <use href="#danger"></use>
              </svg>
              <p className="text-wrap whitespace-normal">
                {errors.categoryid?.message}
              </p>
            </div>
          )}
        </div>
        <div className="child:text-[#696464] dark:child:text-light">
          <p className="font text-lg">
            <span className="font text-lg">زیر دسته بندی محصول :</span>
          </p>

          <select
            className="w-full h-[53px] mt-2.25 bg-[#EFEFEF] dark:bg-[#535353] p-4 pl-10 text-base-mines rounded-sm outline-none"
            {...register("subcategoryid")}
          >
            <option value="-1" defaultChecked>
              {subCategory_data
                ? "انتخاب کنید"
                : "ابتدا دسته بندی خود را انتخاب کنید"}
            </option>
            {subCategory_data?.subcategories?.map((category: CategoryType) => (
              <option key={category._id} value={category._id}>
                {category.title}
              </option>
            ))}
          </select>
          {errors.subcategoryid && (
            <div className="flex items-center gap-2.5 w-full mt-2.5 child:text-[#FF0000] text-xs">
              <svg className="size-5 shrink-0">
                <use href="#danger"></use>
              </svg>
              <p className="text-wrap whitespace-normal">
                {errors.subcategoryid?.message}
              </p>
            </div>
          )}
        </div>
      </div>
      {/* long-dec */}
      <div
        className={`child:text-[#696464] dark:child:text-light col-span-1 md:col-span-2 xl:col-span-3`}
      >
        <p className="my-2 text-lg">
          توضیحات کامل <span className="text-error-400">*</span>
        </p>
        <Controller
          name="decs"
          control={control}
          render={({ field }) => (
            <BlogEditor
              blogContent={field.value}
              setBlogContent={field.onChange}
            />
          )}
        />
        {errors.decs && (
          <div className="flex items-center gap-2.5 w-full mt-2.5 child:text-[#FF0000] text-xs">
            <svg className="size-5 shrink-0">
              <use href="#danger"></use>
            </svg>
            <p className="text-wrap whitespace-normal">
              {errors.decs?.message}
            </p>
          </div>
        )}
      </div>
      {/* uploader */}
      <div>
        <p className="my-2.5 font text-lg child:text-[#696464] dark:child:text-light">
          <span className="font text-lg">عکس محصول :</span>
        </p>
        <label className="flex items-center justify-between w-full sm:w-[180px] py-4 px-6 mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 rounded-md border border-dashed border-[#DDDDDD] dark:border-[#9B9B9B]/50 cursor-pointer">
          <span className="text-[#9B9B9B]">آپلود پیوست</span>
          <input
            className="opacity-0 w-0 h-0"
            type="file"
            multiple
            onChange={handleFileChange}
          />
          <svg className="w-6 h-6 text-[#9B9B9B]">
            <use href="#file"></use>
          </svg>
        </label>
        {isFileError && fileError && (
          <div className="flex items-center gap-2.5 w-full mt-2.5 child:text-[#FF0000] text-xs">
            <svg className="size-5 shrink-0">
              <use href="#danger"></use>
            </svg>
            <p className="text-wrap whitespace-normal">{fileError}</p>
          </div>
        )}
      </div>
      <div className="col-span-1 md:col-span-2 xl:col-span-3 flex flex-wrap gap-2 my-2.5">
        {file?.map((f, index) => (
          <div
            key={index}
            className="relative w-24 h-24 border rounded overflow-hidden"
          >
            <Image
              src={URL.createObjectURL(f)}
              alt={f.name}
              width={600}
              height={600}
              className="w-full h-full object-cover"
            />
            <Button
              className="absolute top-1 right-1 bg-[#FF0000] hover:bg-white hover:text-[#FF0000] text-white w-5 h-5 p-0 m-0 flex items-center justify-center rounded-full"
              onClick={() =>
                setFile((prev) =>
                  prev ? prev.filter((_, i) => i !== index) : null
                )
              }
            >
              <svg>
                <use href="#x-mark"></use>
              </svg>
            </Button>
          </div>
        ))}
      </div>

      {/* cover */}
      <div>
        <p className="font text-lg child:text-[#696464] dark:child:text-light">
          <span className="font text-lg">عکس کاور :</span>
        </p>
        <label className="flex items-center justify-between w-full sm:w-[180px] py-4 px-6 mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 rounded-md border border-dashed border-[#DDDDDD] dark:border-[#9B9B9B]/50 cursor-pointer">
          <span className="text-[#9B9B9B]">آپلود پیوست</span>
          <input
            className="opacity-0 w-0 h-0"
            type="file"
            onChange={handleCoverChange}
          />
          <svg className="w-6 h-6 text-[#9B9B9B]">
            <use href="#file"></use>
          </svg>
        </label>
        {isCoverError && coverError && (
          <div className="flex items-center gap-2.5 w-full mt-2.5 child:text-[#FF0000] text-xs">
            <svg className="size-5 shrink-0">
              <use href="#danger"></use>
            </svg>
            <p className="text-wrap whitespace-normal">{coverError}</p>
          </div>
        )}
      </div>
      <div className="col-span-1 md:col-span-2 xl:col-span-3 flex flex-wrap gap-2 my-2.5">
        {cover?.map((f, index) => (
          <div
            key={index}
            className="relative w-24 h-24 border rounded overflow-hidden"
          >
            <Image
              src={URL.createObjectURL(f)}
              alt={f.name}
              width={96}
              height={96}
              className="w-full h-full object-cover"
            />
            <Button
              className="absolute top-1 right-1 bg-[#FF0000] hover:bg-white hover:text-[#FF0000] text-white w-5 h-5 p-0 m-0 flex items-center justify-center rounded-full"
              onClick={() =>
                setCover((prev) =>
                  prev ? prev.filter((_, i) => i !== index) : null
                )
              }
            >
              <svg>
                <use href="#x-mark"></use>
              </svg>
            </Button>
          </div>
        ))}
      </div>
      {/* file */}
      <div className="mb-3 col-span-3">
        <p className="font text-lg child:text-[#696464] dark:child:text-light">
          <span className="font text-lg">فایل محصول :</span>
        </p>
        <label className="flex items-center justify-between w-full sm:w-[180px] py-4 px-6 mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 rounded-md border border-dashed border-[#DDDDDD] dark:border-[#9B9B9B]/50 cursor-pointer">
          <span className="text-[#9B9B9B]">آپلود فایل</span>
          <input
            className="opacity-0 w-0 h-0"
            type="file"
            onChange={handleProductFileChange}
            ref={productFileInputRef}
          />
          <svg className="w-6 h-6 text-[#9B9B9B]">
            <use href="#file"></use>
          </svg>
        </label>
        {isProductFileError && productFileError && (
          <div className="flex items-center gap-2.5 w-full mt-2.5 child:text-[#FF0000] text-xs">
            <svg className="size-5 shrink-0">
              <use href="#danger"></use>
            </svg>
            <p className="text-wrap whitespace-normal">{productFileError}</p>
          </div>
        )}
        {uploading && (
          <div className="flex items-center gap-2 md:w-[250px] my-4">
            <div className="w-[85%] bg-gray-200 rounded h-2 ">
              <div
                className="bg-secondary-default h-2 rounded transition-all"
                style={{ width: `${uploadProductFileProgress}%` }}
              />
            </div>
            <p className="grow w-5 text-xs">{uploadProductFileProgress} %</p>
          </div>
        )}
        {productFile && (
          <div className="flex items-center justify-between md:w-[250px] mt-2.5 bg-secondary-200 border border-secondary-default text-light px-3 py-2 rounded ">
            <span className="text-sm text-dark-500 truncate">
              {productFileName}
            </span>

            <button
              type="button"
              onClick={removeUploadFileHandler}
              className="text-red-500 hover:text-red-700 text-lg"
            >
              ×
            </button>
          </div>
        )}
      </div>

      <div className="flex flex-col md:flex-row items-start gap-x-10 w-full my-4 child:w-full md:child:w-1/2">
        <div>
          <div>
            <p>ویژگی ها</p>
            {optionFields.map((_, index) => (
              <>
                <div className="flex items-center gap-x-2 gap-y-5">
                  <input
                    className="w-full h-[53px] mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 p-4 text-base-mines rounded-sm"
                    type="text"
                    placeholder="مثال: پشتیبانی خصوصی"
                    {...register(`customOptions.${index}.title` as const)}
                  />
                  <input
                    className="w-full h-[53px] mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 p-4 text-base-mines rounded-sm"
                    type="text"
                    placeholder="مثال 1000000"
                    {...register(`customOptions.${index}.value` as const)}
                  />
                  <span
                    className={`flex-center shrink-0 size-[53px] mt-2.25 ${
                      optionFields.length === 1
                        ? "bg-[#EFEFEF]/70 dark:bg-dark-500/50 text-neutral-150 dark:text-neutral-150/50 cursor-auto"
                        : "bg-[#EFEFEF] dark:bg-dark-500 text-neutral-500 hover:bg-[#FF0000] dark:hover:bg-[#FF0000] hover:text-light"
                    } rounded-9 cursor-pointer transition-all duration-200`}
                    onClick={() => {
                      if (optionFields?.length === 1) return;
                      removeOption(index);
                    }}
                  >
                    <svg className=" size-5">
                      <use href="#trash"></use>
                    </svg>
                  </span>
                </div>
                {errors.customOptions?.[index] && (
                  <div className="flex items-center gap-2.5 w-full mt-2.5 child:text-[#FF0000] text-xs">
                    <svg className="size-5 shrink-0">
                      <use href="#danger"></use>
                    </svg>
                    <p className="text-wrap whitespace-normal">
                      {errors.customOptions?.message ||
                        errors.customOptions?.[index]?.title?.message ||
                        errors.customOptions?.[index]?.value?.message}
                    </p>
                  </div>
                )}
              </>
            ))}
          </div>

          <Button
            type="button"
            className="flex items-center gap-1 w-full h-[53px] my-2  p-2 font-peyda font-bold bg-secondary-200/70 hover:bg-secondary-default hover:text-light text-secondary-default rounded-9 transition-all duration-200"
            onClick={() => addOption({ title: "", value: "" })}
          >
            <svg className="size-5">
              <use href="#plus"></use>
            </svg>
            <p>افزودن ویژگی جدید</p>
          </Button>
        </div>
        <div>
          <div>
            <p>امکانات</p>
            {featureFields.map((_, index) => (
              <>
                <div className="flex items-center gap-x-2 gap-y-5">
                  <input
                    className="w-full h-[53px] mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 p-4 text-base-mines rounded-sm"
                    type="text"
                    placeholder="مثال: حجم"
                    {...register(`features.${index}.name` as const)}
                  />
                  <input
                    className="w-full h-[53px] mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 p-4 text-base-mines rounded-sm"
                    type="text"
                    placeholder="مثال: 500 MB"
                    {...register(`features.${index}.detail` as const)}
                  />
                  <span
                    className={`flex-center shrink-0 size-[53px] mt-2.25  rounded-9 cursor-pointer transition-all duration-200 ${
                      featureFields.length === 1
                        ? "bg-[#EFEFEF]/70 dark:bg-dark-500/50 text-neutral-150 dark:text-neutral-150/50 cursor-auto"
                        : "bg-[#EFEFEF] dark:bg-dark-500 text-neutral-500 hover:bg-[#FF0000] dark:hover:bg-[#FF0000] hover:text-light"
                    }`}
                    onClick={() => {
                      if (featureFields.length === 1) return;
                      removeFeature(index);
                    }}
                  >
                    <svg className=" size-5">
                      <use href="#trash"></use>
                    </svg>
                  </span>
                </div>
                {errors.features?.[index] && (
                  <div className="flex items-center gap-2.5 w-full mt-2.5 child:text-[#FF0000] text-xs">
                    <svg className="size-5 shrink-0">
                      <use href="#danger"></use>
                    </svg>
                    <p className="text-wrap whitespace-normal">
                      {errors.features?.message ||
                        errors.features?.[index]?.name?.message ||
                        errors.features?.[index]?.detail?.message}
                    </p>
                  </div>
                )}
              </>
            ))}
          </div>

          <Button
            type="button"
            className="flex items-center gap-1 w-full h-[53px] my-2  p-2 font-peyda font-bold bg-secondary-200/70 hover:bg-secondary-default hover:text-light text-secondary-default rounded-9 transition-all duration-200"
            onClick={() => addFeature({ name: "", detail: "" })}
          >
            <svg className="size-5">
              <use href="#plus"></use>
            </svg>
            <p>افزودن امکانات جدید</p>
          </Button>
        </div>
      </div>
      <div className="w-full text-left xl:text-left mt-[23px]">
        <Button
          className="hidden sm:inline-block flex-center w-full xl:w-[165px] h-[55px] px-5 bg-secondary-default hover:bg-secondary-default/90 text-base dark:text-light rounded-10"
          type="submit"
          disabled={isPending}
        >
          {isPending ? `در حال آپلود ${uploadProgress} %` : "ایجاد"}
        </Button>
      </div>
    </form>
  );
}
