"use client";
import { ApiError, ApiResponse } from "@/@types/api";
import { CategoryType } from "@/@types/categorey";
import { file } from "@/@types/file";
import CustomToast from "@/components/modules/CustomToast";
import { Button } from "@/components/ui/button";
import { Checkbox } from "@/components/ui/Checkbox";
import { FileSchema } from "@/configs/ValidationSchema";
import { getAdminCategory, getAdminSubCategory } from "@/services/category";
import { updateAdminFile } from "@/services/file";
import { yupResolver } from "@hookform/resolvers/yup";
import { useMutation, useQuery, useQueryClient } from "@tanstack/react-query";
import { AxiosResponse } from "axios";
import dynamic from "next/dynamic";
import Image from "next/image";
import React, { Dispatch, Fragment, SetStateAction, useState } from "react";
import {
  Controller,
  SubmitHandler,
  useFieldArray,
  useForm,
} from "react-hook-form";
import * as yup from "yup";
const BlogEditor = dynamic(() => import("../blogs/blog/BlogEditor"), {
  ssr: false,
});

type InputsType = yup.InferType<typeof FileSchema>;
type inputs = {
  id: number;
  label: string;
  required: boolean;
  inputType: string;
  textarea?: boolean;
  checkbox?: boolean;
  value:
    | "title"
    | "subdecs"
    | "decs"
    | "price"
    | "categoryid"
    | "subcategoryid"
    | "previewlink"
    | "customOptions"
    | "features"
    | "isaccept";
};

export default function UpdateFileForm({
  fileData,
  setIsOpenEditModal,
}: {
  fileData: file | null;
  setIsOpenEditModal: Dispatch<SetStateAction<boolean>>;
}) {
  const {
    register,
    handleSubmit,
    formState: { errors },
    control,
    watch,
  } = useForm<InputsType>({
    resolver: yupResolver(FileSchema),
    defaultValues: {
      title: fileData?.title ?? "",
      subdecs: fileData?.subdecs ?? "",
      decs: fileData?.decs ?? "",
      price: fileData?.price ? String(fileData?.price) : "",
      categoryid: fileData?.categoryid ?? "",
      subcategoryid: fileData?.subcategoryid ?? "",
      previewlink: fileData?.previewlink ?? "",
      isaccept: fileData?.isaccept ?? false,
      features: fileData?.features ?? [{ name: "", detail: "" }],
      customOptions: fileData?.customOptions ?? [{ title: "", value: "" }],
    },
  });

  const {
    fields: featureFields,
    append: addFeature,
    remove: removeFeature,
  } = useFieldArray({
    control,
    name: "features",
  });

  const {
    fields: optionFields,
    append: addOption,
    remove: removeOption,
  } = useFieldArray({
    control,
    name: "customOptions",
  });
  const categoryId = watch("categoryid");

  const { data, isLoading: category_isLoading } = useQuery({
    queryKey: ["admin-category"],
    queryFn: () => getAdminCategory(),
    select: (res) => res.data,
  });

  const { data: subCategory_data, isLoading: subCategory_isLoading } = useQuery(
    {
      queryKey: ["admin-subcategory", categoryId],
      queryFn: () => getAdminSubCategory(String(categoryId)),
      select: (res) => res?.data,
      enabled: !!categoryId,
    }
  );

  const inputs: inputs[] = [
    {
      id: 1,
      label: " عنوان :",
      required: false,
      inputType: "text",
      value: "title",
    },
    {
      id: 4,
      label: "قیمت :",
      required: false,
      inputType: "text",
      value: "price",
    },
    {
      id: 7,
      label: "لینک پیش نمایش :",
      required: false,
      inputType: "text",
      value: "previewlink",
    },
    {
      id: 11,
      label: "محصول فعال",
      required: false,
      inputType: "text",
      checkbox: true,
      value: "isaccept",
    },
    // {
    //   id: 12,
    //   label: "  توضیحات بیشتر :",
    //   required: false,
    //   inputType: "text",
    //   value: "decs",
    //   textarea: true,
    // },
    {
      id: 13,
      label: " توضیحات کوتاه :",
      required: false,
      inputType: "text",
      value: "subdecs",
      textarea: true,
    },
  ];

  const [uploadProgress, setUploadProgress] = useState<number>(0);

  const [fileError, setFileError] = useState<string | null>(null);
  const [file, setFile] = useState<File[] | null>(null);
  const [isFileError, setIsFileError] = useState<boolean>(false);

  const [coverError, setCoverError] = useState<string | null>(null);
  const [cover, setCover] = useState<File[] | null>(null);
  const [isCoverError, setIsCoverError] = useState<boolean>(false);

  const { mutate, isPending } = useMutation<
    AxiosResponse<ApiResponse<string>>,
    ApiError,
    {
      id: string;
      data: FormData;
      onProgress?: (percent: number) => void;
    }
  >({ mutationFn: updateAdminFile });
  const queryClient = useQueryClient();

  // const handleFileChange = (e: React.ChangeEvent<HTMLInputElement>) => {
  //   const fileList = e.target.files;
  //   if (!fileList) return;

  //   setIsFileError(false);
  //   setFileError(null);

  //   const selectedFiles = Array.from(fileList);
  //   setFile(selectedFiles);
  //   e.target.value = "";
  // };

  const MAX_FILES = 10;
  const MAX_FILE_SIZE_MB = 1;

  const handleFileChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const files = e.target.files;
    if (!files) return;

    setIsFileError(false);
    setFileError(null);

    const selectedFiles = Array.from(files);

    // محدودیت تعداد
    if (selectedFiles.length + (file?.length || 0) > MAX_FILES) {
      setIsFileError(true);
      setFileError(`حداکثر ${MAX_FILES} فایل می‌توانید انتخاب کنید`);
      return;
    }

    // محدودیت حجم
    const invalidFile = selectedFiles.find(
      (f) => f.size / 1024 / 1024 > MAX_FILE_SIZE_MB
    );
    if (invalidFile) {
      setIsFileError(true);
      setFileError(
        `حجم فایل‌ها نباید بیشتر از ${MAX_FILE_SIZE_MB} مگابایت باشد`
      );
      return;
    }

    // اضافه کردن فایل‌ها به استیت
    setFile((prev) => (prev ? [...prev, ...selectedFiles] : selectedFiles));
    e.target.value = "";
  };
  const MAX_COVER = 1;

  const handleCoverChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const files = e.target.files;
    if (!files) return;

    setIsCoverError(false);
    setCoverError(null);

    const selectedFiles = Array.from(files);

    // محدودیت تعداد
    if (selectedFiles.length + (cover?.length || 0) > MAX_COVER) {
      setIsCoverError(true);
      setCoverError(`حداکثر ${MAX_COVER} فایل می‌توانید انتخاب کنید`);
      return;
    }

    // محدودیت حجم
    const invalidFile = selectedFiles.find(
      (f) => f.size / 1024 / 1024 > MAX_FILE_SIZE_MB
    );
    if (invalidFile) {
      setIsCoverError(true);
      setCoverError(
        `حجم عکس ها نباید بیشتر از ${MAX_FILE_SIZE_MB} مگابایت باشد`
      );
      return;
    }

    // اضافه کردن عکس ها به استیت
    setCover((prev) => (prev ? [...prev, ...selectedFiles] : selectedFiles));
    e.target.value = "";
  };

  const submitHandler: SubmitHandler<InputsType> = (data) => {
    const formData = new FormData();
    if (file || file !== null) {
      file.forEach((f) => {
        formData.append("file", f);
      });
    }

    const cleanedFeatures =
      data.features?.filter(
        (item) => item.name?.trim() || item.detail?.trim()
      ) || [];

    const cleanedOptions =
      data.customOptions?.filter(
        (item) => item.title?.trim() || item.value?.trim()
      ) || [];

    Object.entries(data).forEach(([key, value]) => {
      if (key === "features") {
        formData.append("features", JSON.stringify(cleanedFeatures));
      } else if (key === "customOptions") {
        formData.append("customOptions", JSON.stringify(cleanedOptions));
      } else {
        formData.append(key, String(value));
      }
    });
    mutate(
      { id: fileData?._id ?? "", data: formData },
      {
        onSuccess: (res) => {
          if (res.status === 200) {
            CustomToast({
              type: "success",
              text: "محصول  با موفقیت ویرایش شد",
            });
            queryClient.invalidateQueries({
              queryKey: ["admin-files"],
            });
            queryClient.invalidateQueries({
              queryKey: ["dev-files"],
            });
          }
          setUploadProgress(0);
          setIsOpenEditModal(false);
        },
        onError: (error) => {
          const statusCode = error?.response?.status;
          if (statusCode === 404) {
            CustomToast({
              type: "error",
              text: "محصول یافت نشد",
            });
          } else {
            CustomToast({
              type: "error",
              text: "خطا در انجام عملیات",
              des: "لطفا دوباره امتحان کنید",
            });
          }
          setUploadProgress(0);
        },
      }
    );
  };

  return (
    <form onSubmit={handleSubmit(submitHandler)}>
      <div className="grid  grid-cols-1 md:grid-cols-2 xl:grid-cols-3 gap-x-7 xl:gap-x-10 gap-y-7 xl:gap-y-[32px] mt-10">
        {inputs.map((input) => (
          <div
            key={input.id}
            className={`child:text-[#696464] dark:child:text-light ${
              input.textarea ? "col-span-1 md:col-span-2 xl:col-span-3" : ""
            }`}
          >
            {!input.checkbox ? (
              <p className="font text-lg">
                {input.label}{" "}
                {input.required ? (
                  <span className="text-error-400">*</span>
                ) : (
                  ""
                )}
              </p>
            ) : null}
            {input.textarea ? (
              <textarea
                className="w-full  mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 p-4 text-base-mines rounded-sm outline-none"
                placeholder="وارد کنید"
                {...register(input.value)}
                rows={4}
              />
            ) : input.checkbox ? (
              <Controller
                name="isaccept"
                control={control}
                render={({ field }) => (
                  <div className="flex items-center gap-2 h-full my-1">
                    <Checkbox
                      className="size-6"
                      checked={field.value}
                      onCheckedChange={field.onChange}
                    />
                    <p className="text-lg">محصول فعال</p>
                  </div>
                )}
              />
            ) : (
              <input
                className="w-full h-[53px] mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 p-4 text-base-mines rounded-sm"
                type={input.inputType}
                placeholder="وارد کنید"
                {...register(input.value)}
              />
            )}

            {errors[input.value] && (
              <div className="flex items-center gap-2.5 w-full mt-2.5 child:text-[#FF0000] text-xs">
                <svg className="size-5 shrink-0">
                  <use href="#danger"></use>
                </svg>
                <p className="text-wrap whitespace-normal">
                  {errors[input.value]?.message}
                </p>
              </div>
            )}
          </div>
        ))}
        {category_isLoading || subCategory_isLoading ? (
          <>
            <div className="child:text-[#696464] dark:child:text-light">
              <p className="font text-lg">
                <span className="font text-lg">دسته بندی محصول :</span>
              </p>

              <select className="w-full h-[53px] mt-2.25 bg-[#EFEFEF] dark:bg-[#535353] p-4 pl-10 text-base-mines rounded-sm outline-none">
                <option defaultChecked>لطفا صبر کنید</option>
              </select>
            </div>
            <div className="child:text-[#696464] dark:child:text-light">
              <p className="font text-lg">
                <span className="font text-lg">زیر دسته بندی محصول :</span>
              </p>

              <select className="w-full h-[53px] mt-2.25 bg-[#EFEFEF] dark:bg-[#535353] p-4 pl-10 text-base-mines rounded-sm outline-none">
                <option defaultChecked>لطفا صبر کنید</option>
              </select>
            </div>
          </>
        ) : (
          <>
            <div className="child:text-[#696464] dark:child:text-light">
              <p className="font text-lg">
                <span className="font text-lg">دسته بندی محصول :</span>
              </p>

              <select
                className="w-full h-[53px] mt-2.25 bg-[#EFEFEF] dark:bg-[#535353] p-4 pl-10 text-base-mines rounded-sm outline-none"
                {...register("categoryid")}
              >
                <option value="-1" defaultChecked>
                  انتخاب کنید
                </option>
                {data?.categorys?.map((category: CategoryType) => (
                  <option key={category._id} value={category._id}>
                    {category.title}
                  </option>
                ))}
              </select>
              {errors.categoryid && (
                <div className="flex items-center gap-2.5 w-full mt-2.5 child:text-[#FF0000] text-xs">
                  <svg className="size-5 shrink-0">
                    <use href="#danger"></use>
                  </svg>
                  <p className="text-wrap whitespace-normal">
                    {errors.categoryid?.message}
                  </p>
                </div>
              )}
            </div>
            <div className="child:text-[#696464] dark:child:text-light">
              <p className="font text-lg">
                <span className="font text-lg">زیر دسته بندی محصول :</span>
              </p>

              <select
                className="w-full h-[53px] mt-2.25 bg-[#EFEFEF] dark:bg-[#535353] p-4 pl-10 text-base-mines rounded-sm outline-none"
                {...register("subcategoryid")}
              >
                <option value="-1" defaultChecked>
                  {subCategory_data
                    ? "انتخاب کنید"
                    : "ابتدا دسته بندی خود را انتخاب کنید"}
                </option>
                {subCategory_data?.subcategories?.map(
                  (category: CategoryType) => (
                    <option key={category._id} value={category._id}>
                      {category.title}
                    </option>
                  )
                )}
              </select>
              {errors.subcategoryid && (
                <div className="flex items-center gap-2.5 w-full mt-2.5 child:text-[#FF0000] text-xs">
                  <svg className="size-5 shrink-0">
                    <use href="#danger"></use>
                  </svg>
                  <p className="text-wrap whitespace-normal">
                    {errors.subcategoryid?.message}
                  </p>
                </div>
              )}
            </div>
          </>
        )}
      </div>

      <div className="col-span-1 md:col-span-2 xl:col-span-3 flex flex-wrap gap-2 my-2.5">
        {file?.map((f, index) => (
          <div
            key={index}
            className="relative w-24 h-24 border rounded overflow-hidden"
          >
            <Image
              src={URL.createObjectURL(f)}
              alt={f.name}
              width={96}
              height={96}
              className="w-full h-full object-cover"
            />
            <Button
              className="absolute top-1 right-1 bg-[#FF0000] hover:bg-white hover:text-[#FF0000] text-white w-5 h-5 p-0 m-0 flex items-center justify-center rounded-full"
              onClick={() =>
                setFile((prev) =>
                  prev ? prev.filter((_, i) => i !== index) : null
                )
              }
            >
              <svg>
                <use href="#x-mark"></use>
              </svg>
            </Button>
          </div>
        ))}
      </div>

      {/* long-dec */}
      <div
        className={`child:text-[#696464] dark:child:text-light col-span-1 md:col-span-2 xl:col-span-3`}
      >
        <p className="my-2 text-lg">
          توضیحات کامل <span className="text-error-400">*</span>
        </p>
        <Controller
          name="decs"
          control={control}
          render={({ field }) => (
            <BlogEditor
              blogContent={field.value}
              setBlogContent={field.onChange}
            />
          )}
        />
        {errors.decs && (
          <div className="flex items-center gap-2.5 w-full mt-2.5 child:text-[#FF0000] text-xs">
            <svg className="size-5 shrink-0">
              <use href="#danger"></use>
            </svg>
            <p className="text-wrap whitespace-normal">
              {errors.decs?.message}
            </p>
          </div>
        )}
      </div>

      <div className="flex flex-col md:flex-row items-start gap-x-10 w-full my-4 child:w-full md:child:w-1/2">
        <div>
          <div>
            <p>ویژگی ها:</p>
            {optionFields.map((item, index) => (
              <Fragment key={(item?.id, index)}>
                <div className="flex items-center gap-x-2 gap-y-5">
                  <input
                    className="w-full h-[53px] mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 p-4 text-base-mines rounded-sm"
                    type="text"
                    placeholder="مثال: پشتیبانی "
                    {...register(`customOptions.${index}.title` as const)}
                  />
                  <input
                    className="w-full h-[53px] mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 p-4 text-base-mines rounded-sm"
                    type="text"
                    placeholder="مثال  1000000"
                    {...register(`customOptions.${index}.value` as const)}
                  />
                  <span
                    className={`flex-center shrink-0 size-[53px] mt-2.25 ${
                      optionFields.length === 1
                        ? "bg-[#EFEFEF]/70 dark:bg-dark-500/50 text-neutral-150 dark:text-neutral-150/50 cursor-auto"
                        : "bg-[#EFEFEF] dark:bg-dark-500 text-neutral-500 hover:bg-[#FF0000] dark:hover:bg-[#FF0000] hover:text-light"
                    } rounded-9 cursor-pointer transition-all duration-200`}
                    onClick={() => {
                      if (optionFields?.length === 1) return;
                      removeOption(index);
                    }}
                  >
                    <svg className=" size-5">
                      <use href="#trash"></use>
                    </svg>
                  </span>
                </div>
                {errors.customOptions?.[index] && (
                  <div className="flex items-center gap-2.5 w-full mt-2.5 child:text-[#FF0000] text-xs">
                    <svg className="size-5 shrink-0">
                      <use href="#danger"></use>
                    </svg>
                    <p className="text-wrap whitespace-normal">
                      {errors.customOptions?.message ||
                        errors.customOptions?.[index]?.title?.message ||
                        errors.customOptions?.[index]?.value?.message}
                    </p>
                  </div>
                )}
              </Fragment>
            ))}
          </div>

          <Button
            type="button"
            className="flex items-center gap-1 w-full h-[53px] my-2  p-2 font-peyda font-bold bg-secondary-200/70 hover:bg-secondary-default hover:text-light text-secondary-default rounded-9 transition-all duration-200"
            onClick={() => addOption({ title: "", value: "" })}
          >
            <svg className="size-5">
              <use href="#plus"></use>
            </svg>
            <p>افزودن ویژگی جدید</p>
          </Button>
        </div>
        <div>
          <div>
            <p>امکانات:</p>
            {featureFields.map((item, index) => (
              <Fragment key={item?.id}>
                <div className="flex items-center gap-x-2 gap-y-5">
                  <input
                    className="w-full h-[53px] mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 p-4 text-base-mines rounded-sm"
                    type="text"
                    placeholder="مثال: حجم"
                    {...register(`features.${index}.name` as const)}
                  />
                  <input
                    className="w-full h-[53px] mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 p-4 text-base-mines rounded-sm"
                    type="text"
                    placeholder="مثال: 500 MB"
                    {...register(`features.${index}.detail` as const)}
                  />
                  <span
                    className={`flex-center shrink-0 size-[53px] mt-2.25  rounded-9 cursor-pointer transition-all duration-200 ${
                      featureFields.length === 1
                        ? "bg-[#EFEFEF]/70 dark:bg-dark-500/50 text-neutral-150 dark:text-neutral-150/50 cursor-auto"
                        : "bg-[#EFEFEF] dark:bg-dark-500 text-neutral-500 hover:bg-[#FF0000] dark:hover:bg-[#FF0000] hover:text-light"
                    }`}
                    onClick={() => {
                      if (featureFields.length === 1) return;
                      removeFeature(index);
                    }}
                  >
                    <svg className=" size-5">
                      <use href="#trash"></use>
                    </svg>
                  </span>
                </div>
                {errors.features?.[index] && (
                  <div className="flex items-center gap-2.5 w-full mt-2.5 child:text-[#FF0000] text-xs">
                    <svg className="size-5 shrink-0">
                      <use href="#danger"></use>
                    </svg>
                    <p className="text-wrap whitespace-normal">
                      {errors.features?.message ||
                        errors.features?.[index]?.name?.message ||
                        errors.features?.[index]?.detail?.message}
                    </p>
                  </div>
                )}
              </Fragment>
            ))}
          </div>

          <Button
            type="button"
            className="flex items-center gap-1 w-full h-[53px] my-2  p-2 font-peyda font-bold bg-secondary-200/70 hover:bg-secondary-default hover:text-light text-secondary-default rounded-9 transition-all duration-200"
            onClick={() => addFeature({ name: "", detail: "" })}
          >
            <svg className="size-5">
              <use href="#plus"></use>
            </svg>
            <p>افزودن امکانات جدید</p>
          </Button>
        </div>
      </div>

      <div className="py-4 border-y border-light-600/20">
        {/* prodct-img */}
        <div className="col-span-1 md:col-span-2 xl:col-span-3 my-2.5">
          <p className="my-2 text-lg">عکس های آپلود شده :</p>
          <div className="flex flex-wrap gap-2 my-2.5">
            {fileData && fileData?.images?.length > 0 ? (
              fileData?.images?.map((img: string, index: number) => (
                <div
                  key={index + img}
                  className="relative w-24 h-24 border rounded overflow-hidden"
                >
                  <Image
                    src={`${process.env.NEXT_PUBLIC_BASE_URL}${img}`}
                    alt={img}
                    width={600}
                    height={600}
                    className="w-full h-full object-cover"
                  />
                  {/* <Button
                  type="button"
                  className="absolute top-1 right-1 bg-[#FF0000] hover:bg-white hover:text-[#FF0000] text-white w-5 h-5 p-0 m-0 flex items-center justify-center rounded-full"
                  onClick={() =>
                    setFile((prev) =>
                      prev ? prev.filter((_, i) => i !== index) : null
                    )
                  }
                >
                  <svg>
                    <use href="#x-mark"></use>
                  </svg>
                </Button> */}
                </div>
              ))
            ) : (
              <p className="text-red-500">دیتایی یافت نشد</p>
            )}
          </div>
        </div>

        {/* upload */}
        <div>
          <p className="font text-lg child:text-[#696464] dark:child:text-light">
            <span className="font text-lg">آپلود عکس جدید :</span>
          </p>
          <label className="flex items-center justify-between w-full sm:w-[180px] py-4 px-6 mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 rounded-md border border-dashed border-[#DDDDDD] dark:border-[#9B9B9B]/50 cursor-pointer">
            <span className="text-[#9B9B9B]">آپلود پیوست</span>
            <input
              className="opacity-0 w-0 h-0"
              type="file"
              multiple
              onChange={handleFileChange}
            />
            <svg className="w-6 h-6 text-[#9B9B9B]">
              <use href="#file"></use>
            </svg>
          </label>
          {isFileError && fileError && (
            <div className="flex items-center gap-2.5 w-full mt-2.5 child:text-[#FF0000] text-xs">
              <svg className="size-5 shrink-0">
                <use href="#danger"></use>
              </svg>
              <p className="text-wrap whitespace-normal">{fileError}</p>
            </div>
          )}
        </div>
        <div className="col-span-1 md:col-span-2 xl:col-span-3 flex flex-wrap gap-2 my-2.5">
          {file?.map((f, index) => (
            <div
              key={index}
              className="relative w-24 h-24 border rounded overflow-hidden"
            >
              <Image
                src={URL.createObjectURL(f)}
                alt={f.name}
                width={96}
                height={96}
                className="w-full h-full object-cover"
              />
              <Button
                className="absolute top-1 right-1 bg-[#FF0000] hover:bg-white hover:text-[#FF0000] text-white w-5 h-5 p-0 m-0 flex items-center justify-center rounded-full"
                onClick={() =>
                  setFile((prev) =>
                    prev ? prev.filter((_, i) => i !== index) : null
                  )
                }
              >
                <svg>
                  <use href="#x-mark"></use>
                </svg>
              </Button>
            </div>
          ))}
        </div>
      </div>

      {/* cover-img */}
      <div className="col-span-1 md:col-span-2 xl:col-span-3 my-2.5">
        <p className="my-2 text-lg">عکس کاور آپلود شده :</p>
        <div className="flex flex-wrap gap-2 my-2.5">
          {fileData && fileData?.cover ? (
            <div className="relative w-24 h-24 border rounded overflow-hidden">
              <Image
                src={`${process.env.NEXT_PUBLIC_BASE_URL}${fileData?.cover}`}
                alt={fileData?.title}
                width={600}
                height={600}
                className="w-full h-full object-cover"
              />
            </div>
          ) : (
            <div className="flex items-center gap-1">
              <svg className="size-4 text-error-400">
                <use href="#danger"></use>
              </svg>
              <p className="text-red-500">عکسی آپلود نشده</p>
            </div>
          )}
        </div>
      </div>

      {/* cover */}
      <div>
        <p className="font text-lg child:text-[#696464] dark:child:text-light">
          <span className="font text-lg">آپلود عکس کاور جدید :</span>
        </p>
        <label className="flex items-center justify-between w-full sm:w-[180px] py-4 px-6 mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 rounded-md border border-dashed border-[#DDDDDD] dark:border-[#9B9B9B]/50 cursor-pointer">
          <span className="text-[#9B9B9B]">آپلود پیوست</span>
          <input
            className="opacity-0 w-0 h-0"
            type="file"
            onChange={handleCoverChange}
          />
          <svg className="w-6 h-6 text-[#9B9B9B]">
            <use href="#file"></use>
          </svg>
        </label>
        {isCoverError && coverError && (
          <div className="flex items-center gap-2.5 w-full mt-2.5 child:text-[#FF0000] text-xs">
            <svg className="size-5 shrink-0">
              <use href="#danger"></use>
            </svg>
            <p className="text-wrap whitespace-normal">{coverError}</p>
          </div>
        )}
      </div>
      <div className="col-span-1 md:col-span-2 xl:col-span-3 flex flex-wrap gap-2 my-2.5">
        {cover?.map((f, index) => (
          <div
            key={index}
            className="relative w-24 h-24 border rounded overflow-hidden"
          >
            <Image
              src={URL.createObjectURL(f)}
              alt={f.name}
              width={96}
              height={96}
              className="w-full h-full object-cover"
            />
            <Button
              className="absolute top-1 right-1 bg-[#FF0000] hover:bg-white hover:text-[#FF0000] text-white w-5 h-5 p-0 m-0 flex items-center justify-center rounded-full"
              onClick={() =>
                setCover((prev) =>
                  prev ? prev.filter((_, i) => i !== index) : null
                )
              }
            >
              <svg>
                <use href="#x-mark"></use>
              </svg>
            </Button>
          </div>
        ))}
      </div>

      <div className="flex gap-2 items-center justify-end w-full text-left xl:text-left mt-[23px]">
        <Button
          className="hidden sm:inline-block flex-center w-full xl:w-[165px] h-[55px] px-5 bg-error-400 hover:bg-error-400/90 text-base dark:text-light rounded-10"
          onClick={() => setIsOpenEditModal(false)}
        >
          انصراف
        </Button>
        <Button
          className="hidden sm:inline-block flex-center w-full xl:w-[165px] h-[55px] px-5 bg-secondary-default hover:bg-secondary-default/90 text-base dark:text-light rounded-10"
          type="submit"
          disabled={isPending}
        >
          {isPending ? `در حال آپلود ${uploadProgress} %` : "ویرایش"}
        </Button>
      </div>
    </form>
  );
}
