import { ApiError, ApiResponse } from "@/@types/api";
import CustomToast from "@/components/modules/CustomToast";
import { Button } from "@/components/ui/button";
import { deleteNotification } from "@/services/notification";
import { useMutation, useQueryClient } from "@tanstack/react-query";
import { AxiosResponse } from "axios";
import React, { Dispatch, SetStateAction } from "react";

type DeleteModalProps = {
  setIsOpenDeleteModal: Dispatch<SetStateAction<boolean>>;
  notifyId: string;
};

export default function DeleteModal({
  setIsOpenDeleteModal,
  notifyId,
}: DeleteModalProps) {
  const queryClient = useQueryClient();

  const { mutate, isPending } = useMutation<
    AxiosResponse<ApiResponse<string>>,
    ApiError,
    string
  >({
    mutationFn: deleteNotification,
  });

  const deleteHandler = () => {
    mutate(notifyId, {
      onSuccess: (res) => {
        if (res.status === 200) {
          CustomToast({
            type: "success",
            text: "نوتفیکیشن با موفقیت حذف شد",
          });
        }
        queryClient.invalidateQueries({ queryKey: ["admin-notify"] });

        setIsOpenDeleteModal(false);
      },
      onError: (error) => {
        const statusCode = error?.response?.status;
        if (statusCode === 404) {
          CustomToast({
            type: "error",
            text: "نوتفیکیشن یافت نشد",
          });
        } else {
          CustomToast({
            type: "error",
            text: "خطا در انجام عملیات",
            des: "لطفا دوباره امتحان کنید",
          });
        }
      },
    });
  };

  return (
    <div className="fixed inset-0 flex items-center justify-center mx-auto w-screen h-screen overflow-hidden animate-fadeIn">
      <div className="w-[350px] md:w-[400px]w-[350px] md:w-[600px] p-3 bg-light dark:bg-dark-400 rounded-10 z-50">
        <div>
          <p className="font-bold text-xl text-[#001A46] dark:text-light">
            حذف نوتفیکشن
          </p>
          <div className="flex items-center mt-4">
            <span className="w-[25%] h-0.5 bg-secondary-default dark:bg-primary-200"></span>
            <span className="w-[75%] h-0.5 bg-dark/[7%] dark:bg-[#4A4A4A]"></span>
          </div>
        </div>
        <div>
          <p className="font-peyda text-lg font-bold my-4 text-center">
            آیا از حذف نوتفیکشن مطمئن هستید؟
          </p>
        </div>
        <div className="flex items-center justify-center w-full gap-x-2 mt-[23px]">
          <Button
            className="hidden sm:inline-flex flex-center py-6 px-5 bg-secondary-default hover:bg-secondary-default/90 text-base dark:text-light rounded-10"
            disabled={isPending}
            onClick={deleteHandler}
          >
            بله
          </Button>
          <Button
            className="hidden sm:inline-flex flex-center py-6 px-5 bg-error-400 hover:bg-error-400/90 text-base dark:text-light rounded-10"
            onClick={() => setIsOpenDeleteModal(false)}
          >
            خیر
          </Button>
        </div>
      </div>
      <div
        className="absolute w-full h-full z-40"
        onClick={() => {
          setIsOpenDeleteModal(false);
        }}
      ></div>
    </div>
  );
}
