"use client";
import { ApiError, ApiResponse } from "@/@types/api";
import CustomToast from "@/components/modules/CustomToast";
import { Button } from "@/components/ui/button";
import MultiselectBox from "@/components/ui/MultiselectBox";
import { addNotificationSchema } from "@/configs/ValidationSchema";
import { addNotification } from "@/services/notification";
import { getUsers } from "@/services/users";
import { yupResolver } from "@hookform/resolvers/yup";
import { useMutation, useQuery, useQueryClient } from "@tanstack/react-query";
import { AxiosResponse } from "axios";
import React, { useState } from "react";
import { useForm } from "react-hook-form";
import * as yup from "yup";

type inputs = {
  label: string;
  required: boolean;
  inputType: string;
  textarea?: boolean;
  checkbox?: boolean;
  value: "title" | "description";
};

type InputsType = yup.InferType<typeof addNotificationSchema>;

export default function NotificationForm() {
  const [selectedUsers, setSelectedUsers] = useState<string[]>([]);

  const queryClient = useQueryClient();

  const {
    register,
    handleSubmit,
    formState: { errors },
    reset,
  } = useForm<InputsType>({
    resolver: yupResolver(addNotificationSchema),
    defaultValues: {
      title: "",
      description: "",
    },
  });

  const inputs: inputs[] = [
    {
      label: " عنوان :",
      required: true,
      inputType: "text",
      value: "title",
    },
    {
      label: " توضیحات :",
      required: true,
      inputType: "text",
      value: "description",
      textarea: true,
    },
  ];
  const { data: usersData, isLoading } = useQuery({
    queryKey: ["admin-users"],
    queryFn: getUsers,
    select: (res) => res.data,
  });

  const { mutate, isPending } = useMutation<
    AxiosResponse<ApiResponse<string>>,
    ApiError,
    {
      title: string;
      description: string;
      member: "all" | "user";
      userid: string[] | "";
    }
  >({ mutationFn: addNotification });

  const submitHandler = (data: InputsType) => {
    if (selectedUsers?.length <= 0)
      return CustomToast({
        type: "error",
        text: "کاربران را برای ارسال نوتفیکیشن انتخاب کنید",
      });
    const notifyData: {
      title: string;
      description: string;
      member: "all" | "user";
      userid: string[] | "";
    } = {
      ...data,
      userid: selectedUsers?.includes("all") ? "" : selectedUsers,
      member: selectedUsers?.includes("all") ? "all" : "user",
    };
    mutate(notifyData, {
      onSuccess: () => {
        CustomToast({
          type: "success",
          text: "نوتفیکیشن جدید با موفقیت ایجاد شد",
        });
        queryClient.invalidateQueries({ queryKey: ["admin-notify"] });
        reset({
          title: "",
          description: "",
        });
        setSelectedUsers([]);
      },
      onError: (error) => {
        const statusCode = error?.response?.status;
        const msg = (error?.response?.data as { message?: string })?.message;
        if (statusCode === 400) {
          CustomToast({
            type: "error",
            text: msg || "فیلدها اشتباه است",
          });
        } else {
          CustomToast({
            type: "error",
            text: "خطا در انجام عملیات",
            des: "لطفا دوباره امتحان کنید",
          });
        }
      },
    });
  };

  return (
    <form onSubmit={handleSubmit(submitHandler)}>
      <div className="grid  grid-cols-1 md:grid-cols-2 xl:grid-cols-3 gap-x-7 xl:gap-x-10 gap-y-7 xl:gap-y-[32px] mt-10">
        {inputs.map((input) => (
          <div
            key={input.value}
            className={`child:text-[#696464] dark:child:text-light ${
              input.textarea ? "col-span-1 md:col-span-2 xl:col-span-3" : ""
            }`}
          >
            {!input.checkbox ? (
              <p className="font text-lg">
                {input.label}{" "}
                {input.required ? (
                  <span className="text-error-400">*</span>
                ) : (
                  ""
                )}
              </p>
            ) : null}
            {input.textarea ? (
              <textarea
                className="w-full  mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 p-4 text-base-mines rounded-sm outline-none"
                placeholder="وارد کنید"
                {...register(input.value)}
                rows={4}
              />
            ) : (
              <input
                className="w-full h-[53px] mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 p-4 text-base-mines rounded-sm"
                type={input.inputType}
                placeholder="وارد کنید"
                {...register(input.value)}
              />
            )}
            {errors[input.value] && (
              <div className="flex items-center gap-2.5 w-full mt-2.5 child:text-[#FF0000] text-xs">
                <svg className="size-5 shrink-0">
                  <use href="#danger"></use>
                </svg>
                <p className="text-wrap whitespace-normal">
                  {errors[input.value]?.message}
                </p>
              </div>
            )}
          </div>
        ))}
      </div>
      <div className="mt-2.5 child:text-[#696464] dark:child:text-light">
        <p className="font text-lg">
          <span className="font text-lg">کاربران :</span>
        </p>
      </div>

      {isLoading ? (
        <div className="flex items-center justify-center gap-6 p-5 w-full h-12 rounded-9 bg-neutral-150/20 animate-pulse"></div>
      ) : (
        <MultiselectBox
          users={usersData?.users}
          selectedUsers={selectedUsers}
          setSelectedUsers={setSelectedUsers}
        />
      )}
      <div className="w-full text-left xl:text-left mt-[23px]">
        <Button
          className="hidden sm:inline-block flex-center w-full xl:w-[165px] h-[55px] px-5 bg-secondary-default hover:bg-secondary-default/90 text-base dark:text-light rounded-10"
          type="submit"
          disabled={isPending}
        >
          ایجاد
        </Button>
      </div>
    </form>
  );
}
