import React, { Dispatch, SetStateAction, useEffect, useState } from "react";

import { ApiError, ApiResponse } from "@/@types/api";
import CustomToast from "@/components/modules/CustomToast";
import { Button } from "@/components/ui/button";
import { useMutation, useQuery, useQueryClient } from "@tanstack/react-query";
import { AxiosResponse } from "axios";
import { getSubCategoryById, updateSubCategory } from "@/services/category";
import { categorySchema } from "@/configs/ValidationSchema";
import { useForm } from "react-hook-form";
import { yupResolver } from "@hookform/resolvers/yup";

type editModalProps = {
  setIsOpenEditModal: Dispatch<SetStateAction<boolean>>;
  categoryId: string;
  keyValue?: string[];
  subCategoryId?: string;
};
type InputsType = {
  title: string;
  href: string;
};
type inputs = {
  id: number;
  label: string;
  required: boolean;
  inputType: string;
  textarea?: boolean;
  checkbox?: boolean;
  value: "title" | "href";
};
export default function EditModal({
  setIsOpenEditModal,
  categoryId,
  subCategoryId,
  keyValue,
}: editModalProps) {
  const queryClient = useQueryClient();

  const {
    register,
    handleSubmit,
    formState: { errors },
    reset,
  } = useForm<InputsType>({
    resolver: yupResolver(categorySchema),
    defaultValues: {
      title: "",
      href: "",
    },
  });

  const [fileError, setFileError] = useState<string | null>(null);
  const [file, setFile] = useState<File[]>([]);
  const [isFileError, setIsFileError] = useState<boolean>(false);

  const inputs: inputs[] = [
    {
      id: 1,
      label: " عنوان :",
      required: true,
      inputType: "text",
      value: "title",
    },
    {
      id: 2,
      label: "مسیر (url) :",
      required: true,
      inputType: "text",
      value: "href",
    },
  ];

  const { mutate: subCategory_mutate, isPending: subCategory_isPending } =
    useMutation<
      AxiosResponse<ApiResponse<string>>,
      ApiError,
      {
        id: string;
        data: {
          categoryId: string;
          subId: string;
          title: string;
          href: string;
        };
      }
    >({
      mutationFn: updateSubCategory,
    });

  const { data: subCategory } = useQuery({
    queryKey: ["get-subcategory", subCategoryId],
    queryFn: () => getSubCategoryById(subCategoryId ?? "", categoryId),
    select: (res) => res.data,
  });
  useEffect(() => {
    if (subCategory?.subcategories) {
      reset({
        title: subCategory.subcategories.title,
        href: subCategory.subcategories.href,
      });
    }
  }, [subCategory, reset]);

  const handleFileChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const fileList = e.target.files;
    if (!fileList) return;
    const allowedTypes = ["image/jpeg", "image/png", "image/webp"];
    const selectedFiles = Array.from(fileList);
    const invalidFiles = selectedFiles.filter(
      (file) => !allowedTypes.includes(file.type)
    );

    if (invalidFiles.length > 0) {
      setFileError("فقط فایل‌های تصویری با فرمت JPG, PNG یا WEBP مجاز هستند");
      setIsFileError(true);
      setFile([]);
      e.target.value = "";
      return;
    }

    setFileError(null);
    setIsFileError(false);
    setFile(selectedFiles);
    e.target.value = "";
  };

  const submitHandler = (data: InputsType) => {
    const formData = new FormData();

    Object.entries(data).forEach(([key, value]) => {
      formData.append(key, value as string);
    });

    file.forEach((f) => {
      formData.append("img", f);
    });
    formData.append("id", categoryId);
    subCategory_mutate(
      {
        id: categoryId,
        data: {
          categoryId: categoryId,
          href: data?.href,
          subId: subCategoryId ?? "",
          title: data?.title,
        },
      },
      {
        onSuccess: (res) => {
          if (res.status === 200) {
            CustomToast({
              type: "success",
              text: "زیر دسته بندی جدید با موفقیت ویرایش شد",
            });
          }
          queryClient.invalidateQueries({ queryKey: keyValue });
          queryClient.invalidateQueries({
            queryKey: ["get-subcategory", subCategoryId],
          });

          reset({
            title: "",
            href: "",
          });
          setFile([]);
          setIsOpenEditModal(false);
        },
        onError: (error) => {
          const statusCode = error?.response?.status;
          if (statusCode === 400) {
            CustomToast({
              type: "error",
              text: "ورودی معتبر نمی باشد",
            });
          } else if (statusCode === 401) {
            CustomToast({
              type: "error",
              text: "عدم دسترسی به ارسال دیتا",
            });
          } else {
            CustomToast({
              type: "error",
              text: "خطا در انجام عملیات",
              des: "لطفا دوباره امتحان کنید",
            });
          }
        },
      }
    );
  };

  return (
    <div className="fixed inset-0 flex items-center justify-center mx-auto w-screen h-screen overflow-hidden animate-fadeIn">
      <div className="w-[350px] md:w-[400px]w-[350px] md:w-[600px] p-3 bg-light dark:bg-dark-400 rounded-10 z-50">
        <div>
          <p className="font-bold text-xl text-[#001A46] dark:text-light">
            ویرایش دسته‌بندی
          </p>
          <div className="flex items-center mt-4">
            <span className="w-[25%] h-0.5 bg-secondary-default dark:bg-primary-200"></span>
            <span className="w-[75%] h-0.5 bg-dark/[7%] dark:bg-[#4A4A4A]"></span>
          </div>
        </div>
        <form onSubmit={handleSubmit(submitHandler)}>
          <div className="grid  grid-cols-1 gap-y-5 mt-5">
            {inputs.map((input) => (
              <div
                key={input.id}
                className={`child:text-[#696464] dark:child:text-light ${
                  input.textarea ? "col-span-1 md:col-span-2 xl:col-span-3" : ""
                }`}
              >
                <p className="font text-lg">
                  {input.label}{" "}
                  {input.required ? (
                    <span className="text-error-400">*</span>
                  ) : (
                    ""
                  )}
                </p>

                <input
                  className="w-full h-[53px] mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 p-4 text-base-mines rounded-sm"
                  type={input.inputType}
                  placeholder="وارد کنید"
                  {...register(input.value)}
                />

                {errors[input.value] && (
                  <div className="flex items-center gap-2.5 w-full mt-2.5 child:text-[#FF0000] text-xs">
                    <svg className="size-5 shrink-0">
                      <use href="#danger"></use>
                    </svg>
                    <p className="text-wrap whitespace-normal">
                      {errors[input.value]?.message}
                    </p>
                  </div>
                )}
              </div>
            ))}
            <div>
              <p className="font text-lg child:text-[#696464] dark:child:text-light">
                <span className="font text-lg">فایل :</span>
              </p>
              <label className="flex items-center justify-between w-full py-4 px-6 mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 rounded-md border border-dashed border-[#DDDDDD] dark:border-[#9B9B9B]/50 cursor-pointer">
                <span className="text-[#9B9B9B]">آپلود پیوست</span>
                <input
                  className="opacity-0 w-0 h-0"
                  type="file"
                  accept="image/*"
                  onChange={handleFileChange}
                />
                <svg className="w-6 h-6 text-[#9B9B9B]">
                  <use href="#file"></use>
                </svg>
              </label>
              {isFileError && fileError && (
                <div className="flex items-center gap-2.5 w-full mt-2.5 child:text-[#FF0000] text-xs">
                  <svg className="size-5 shrink-0">
                    <use href="#danger"></use>
                  </svg>
                  <p className="text-wrap whitespace-normal">{fileError}</p>
                </div>
              )}
            </div>
          </div>
          <div className="flex items-center justify-center w-full gap-x-2 mt-[23px]">
            <Button
              className="hidden sm:inline-flex flex-center py-6 px-5 bg-error-400 hover:bg-error-400/90 text-base dark:text-light rounded-10"
              onClick={() => setIsOpenEditModal(false)}
            >
              لغو
            </Button>
            <Button
              className="hidden sm:inline-flex flex-center py-6 px-5 bg-secondary-default hover:bg-secondary-default/90 text-base dark:text-light rounded-10"
              type="submit"
              disabled={subCategory_isPending}
            >
              ارسال
            </Button>
          </div>
        </form>
      </div>

      <div
        className="absolute w-full h-full z-40"
        onClick={() => {
          setIsOpenEditModal(false);
        }}
      ></div>
    </div>
  );
}
