"use client";

import React from "react";

import { useMutation, useQuery, useQueryClient } from "@tanstack/react-query";
import CustomToast from "@/components/modules/CustomToast";
import { ApiError, ApiResponse } from "@/@types/api";
import { AxiosResponse } from "axios";
import LoadingBox from "@/components/modules/LoadingBox";
import { getTicketsAdmin, updateTicketsStatus } from "@/services/tickets";
import { adminTickets } from "@/@types/tickets";
import Link from "next/link";
import { Button } from "@/components/ui/button";

export default function TicketTable() {
  const queryClient = useQueryClient();

  const { data: tickets, isLoading } = useQuery({
    queryKey: ["admin-tickets"],
    queryFn: getTicketsAdmin,
    select: (res) => res.data,
  });

  const { mutate } = useMutation<
    AxiosResponse<ApiResponse<string>>,
    ApiError,
    string
  >({
    mutationFn: updateTicketsStatus,
  });

  const changeStatusHandler = (id: string) => {
    mutate(id, {
      onSuccess: () => {
        CustomToast({
          type: "success",
          text: " وضعیت تیکت با موفقیت تغییر کرد",
        });
        queryClient.invalidateQueries({ queryKey: ["admin-tickets"] });
      },
      onError: (error) => {
        const statusCode = error?.response?.status;
        if (statusCode === 404) {
          CustomToast({
            type: "error",
            text: "متأسفانه تیکت یافت نشد",
          });
        } else if (statusCode === 400) {
          CustomToast({
            type: "error",
            text: "شناسه نامعتبر",
          });
        } else {
          CustomToast({
            type: "error",
            text: "خطا در انجام عملیات",
            des: "لطفا دوباره امتحان کنید",
          });
        }
      },
    });
  };

  if (isLoading) return <LoadingBox />;
  return (
    <table className="w-full mt-7">
      <thead>
        <tr className="child:text-right child:text-[#454545] dark:child:text-light/80 child:text-sm">
          <th className="pb-3.25 border-b border-dark/[7%] dark:border-light/[6%]">
            موضوع تیکت
          </th>
          <th className="pb-3.25 !text-center border-b border-dark/[7%] dark:border-light/[6%]">
            دسته بندی
          </th>
          <th className="pb-3.25 !text-center border-b border-dark/[7%] dark:border-light/[6%]">
            تاریخ ثبت
          </th>
          <th className="pb-3.25 !text-center border-b border-dark/[7%] dark:border-light/[6%]">
            تاریخ آخرین تغیرات
          </th>
          <th className="pb-3.25 !text-center border-b border-dark/[7%] dark:border-light/[6%]">
            وضعیت
          </th>
          <th className="pb-3.25 !text-center border-b border-dark/[7%] dark:border-light/[6%]">
            تغییر وضعیت
          </th>
          <th className="pb-3.25 !text-center border-b border-dark/[7%] dark:border-light/[6%]">
            عملیات
          </th>
        </tr>
      </thead>
      {tickets ? (
        <tbody className="divide-y divide-dark/[7%] dark:divide-light/[6%]">
          {tickets.data?.map((ticket: adminTickets) => (
            <tr
              key={ticket._id}
              className=" child:py-[30px] child:px-2 child:text-right child:grow child:min-w-[150px] lg:child:w-[15%]"
            >
              <td>
                <div className="flex items-center gap-2 xl:gap-[18px]">
                  <p className="text-[#575757] dark:text-light/80 text-sm">
                    {ticket.title}
                  </p>
                </div>
              </td>
              <td>
                <p className="text-[#575757] dark:text-light/80 text-base text-center">
                  {ticket?.category === "1"
                    ? "پشتیبانی محصول"
                    : ticket?.category}
                </p>
              </td>
              <td>
                <p className="text-[#575757] dark:text-light/80 text-base text-center">
                  {new Date(ticket.date).toLocaleDateString("fa-IR")}
                </p>
              </td>
              <td>
                <p className="text-[#575757] dark:text-light/80 text-base text-center">
                  {new Date(ticket.lastdate).toLocaleDateString("fa-IR")}
                </p>
              </td>
              <td>
                <p
                  className={`${
                    ticket.status === "open"
                      ? "text-[#00B074]"
                      : "text-[#FF0000]"
                  } text-center text-base`}
                >
                  {ticket.status === "open" ? "باز" : "بسته شده"}
                </p>
              </td>
              <td className="">
                <Button
                  className={`flex-center p-2.5 duration-150 text-white text-xs mx-auto rounded-md ${
                    ticket.status === "open"
                      ? "bg-[#FF0000] hover:bg-[#FF0000]/90"
                      : "bg-[#00B074] hover:bg-[#00B074]/90"
                  }`}
                  onClick={() => changeStatusHandler(ticket._id)}
                >
                  {ticket.status === "open" ? "بستن" : "باز کردن"}
                </Button>
              </td>
              <td className="flex-center">
                <Link
                  className="inline-block mx-auto p-2.5 bg-secondary-default hover:bg-secondary-default/90 duration-150 child:text-white text-xs rounded-md"
                  href={`/admin-panel/tickets/${ticket._id}`}
                >
                  <p>مشاهده</p>
                </Link>
              </td>
            </tr>
          ))}
        </tbody>
      ) : (
        <tbody>
          <tr>
            <td colSpan={4} className="text-center py-10 text-[#EE404C]">
              دیتایی یافت نشد
            </td>
          </tr>
        </tbody>
      )}
    </table>
  );
}
