"use client";
import { ApiError, ApiResponse } from "@/@types/api";
import { WithdrawalType } from "@/@types/withdrawal";
import CustomToast from "@/components/modules/CustomToast";
import { Button } from "@/components/ui/button";
import { getAdminWithdrawal, updateWithdrawal } from "@/services/withdrawal";
import { useMutation, useQuery, useQueryClient } from "@tanstack/react-query";
import { AxiosError, AxiosResponse } from "axios";
import React, { useState } from "react";

export default function WithdrawalTable() {
  const [withdrawalId, setWithdrawalId] = useState("");
  const queryClient = useQueryClient();
  const { data: withdrawals } = useQuery({
    queryKey: ["admin-withdrawals"],
    queryFn: getAdminWithdrawal,
    select: (res) => res.data.data,
  });
  const { mutate: withdrawal_mutate } = useMutation<
    AxiosResponse<ApiResponse<string>>,
    ApiError,
    string
  >({
    mutationFn: updateWithdrawal,
  });

  const updateHandler = (id: string) => {
    withdrawal_mutate(id, {
      onSuccess: () => {
        CustomToast({
          type: "success",
          text: "وضعیت درخواست تغییر یافت",
        });
        setWithdrawalId("");
        queryClient.invalidateQueries({ queryKey: ["admin-withdrawals"] });
      },
      onError: (error) => {
        const err = error as AxiosError;
        const statusCode = err?.response?.status;
        if (statusCode === 400) {
          CustomToast({
            type: "error",
            text: "شناسه نامعتبر",
          });
        } else if (statusCode === 404) {
          CustomToast({
            type: "error",
            text: "درخواست برداشت یافت نشد",
          });
        } else {
          CustomToast({
            type: "error",
            text: "خطا در انجام عملیات",
            des: "لطفا دوباره امتحان کنید",
          });
        }
        setWithdrawalId("");
      },
    });
  };
  return (
    <table className="w-full mt-7">
      <thead>
        <tr className="child:text-right child:text-[#454545] dark:child:text-light/80 child:text-sm">
          <th className="pb-3.25 !text-center border-b border-dark/[7%] dark:border-light/[6%]">
            ردیف
          </th>
          <th className="pb-3.25 !text-center border-b border-dark/[7%] dark:border-light/[6%]">
            تاریخ ثبت
          </th>
          <th className="pb-3.25 !text-center border-b border-dark/[7%] dark:border-light/[6%]">
            مبلغ
          </th>
          <th className="pb-3.25 !text-center border-b border-dark/[7%] dark:border-light/[6%]">
            وضعیت
          </th>
          <th className="pb-3.25 !text-center border-b border-dark/[7%] dark:border-light/[6%]">
            تغییر وضعیت
          </th>
        </tr>
      </thead>
      <tbody className="divide-y divide-dark/[7%] dark:divide-light/[6%]">
        {withdrawals ? (
          withdrawals?.map((withdrawal: WithdrawalType, index: number) => (
            <tr
              key={withdrawal._id}
              className=" child:py-[30px] child:px-2 child:text-right child:grow child:min-w-[200px] lg:child:w-[20%]"
            >
              <td>
                <p className="text-[#575757] dark:text-light/80 text-base text-center">
                  {index + 1}
                </p>
              </td>
              <td>
                <p className="text-[#575757] dark:text-light/80 text-base text-center">
                  {new Date(withdrawal.updatedAt).toLocaleDateString("fa-IR")}
                </p>
              </td>
              <td>
                <p className="text-[#575757] dark:text-light/80 text-base text-center">
                  {withdrawal?.amount?.toLocaleString()}
                </p>
              </td>
              <td>
                <p
                  className={`${
                    withdrawal.status ? "text-[#00B074]" : "text-[#FF0000]"
                  } text-center text-base`}
                >
                  {withdrawal.status ? "تایید شده" : "تایید نشده / در حال برسی"}
                </p>
              </td>
              <td className="flex-center !w-full">
                <Button
                  className={`inline-block w-20 mx-auto p-2.5 duration-150 child:text-white text-xs rounded-md ${
                    withdrawal?.status
                      ? "bg-[#FF0000] hover:bg-[#FF0000]/90"
                      : "bg-[#00B074] hover:bg-[#00B074]/90"
                  }`}
                  disabled={withdrawal?._id === withdrawalId}
                  onClick={() => {
                    updateHandler(withdrawal?._id);
                    setWithdrawalId(withdrawal?._id);
                  }}
                >
                  {withdrawal?.status ? "رد" : "تایید"}
                </Button>
              </td>
            </tr>
          ))
        ) : (
          <tbody>
            <tr>
              <td colSpan={4} className="text-center py-10 text-[#EE404C]">
                دیتایی یافت نشد
              </td>
            </tr>
          </tbody>
        )}
      </tbody>
    </table>
  );
}
