"use client";
import { ApiError, ApiResponse } from "@/@types/api";
import { ContactFormType } from "@/@types/contact";
import CustomToast from "@/components/modules/CustomToast";
import { Button } from "@/components/ui/button";
import { contactFormSchema } from "@/configs/ValidationSchema";
import { addContactMessage } from "@/services/contact";
import { yupResolver } from "@hookform/resolvers/yup";
import { useMutation } from "@tanstack/react-query";
import { AxiosResponse } from "axios";
import React from "react";
import { useForm } from "react-hook-form";

type input = {
  id: number;
  label: string;
  required: boolean;
  inputType: string;
  value: "fullname" | "phone" | "email" | "department" | "message";
  textarea?: boolean;
  ltr?: boolean;
  leftDir?: boolean;
};

export default function ContactFormLeft() {
  const {
    register,
    handleSubmit,
    reset,
    formState: { errors },
  } = useForm<ContactFormType>({
    resolver: yupResolver(contactFormSchema),
    defaultValues: {
      fullname: "",
      phone: "",
      email: "",
      department: "",
      message: "",
    },
  });

  const inputs: input[] = [
    {
      id: 1,
      label: " نام و نام خانوادگی :",
      required: true,
      inputType: "text",
      value: "fullname",
    },
    {
      id: 2,
      label: " ایمیل :",
      required: true,
      inputType: "text",
      value: "email",
    },
    {
      id: 3,
      label: "موبایل :",
      required: true,
      inputType: "text",
      value: "phone",
    },
    {
      id: 4,
      label: "واحد مورد نظر :",
      required: true,
      inputType: "text",
      value: "department",
    },
    {
      id: 5,
      label: " متن شما :",
      required: true,
      inputType: "text",
      value: "message",
      textarea: true,
    },
  ];

  const { mutate, isPending } = useMutation<
    AxiosResponse<ApiResponse<string>>,
    ApiError,
    ContactFormType
  >({ mutationFn: addContactMessage });

  const submitHandler = (data: ContactFormType) => {
    mutate(data, {
      onSuccess: (res) => {
        if (res.status === 201) {
          CustomToast({
            type: "success",
            text: "پیام شما با موفقیت ارسال شد",
            des: "پس از بررسی پیام شما توسط کارشناسان، با شما تماس گرفته خواهد شد.",
          });
          reset({
            fullname: "",
            phone: "",
            email: "",
            department: "",
            message: "",
          });
        }
      },
      onError: (error) => {
        const statusCode = error?.response?.status;
        if (statusCode === 400) {
          CustomToast({
            type: "error",
            text: "	فیلدهای الزامی پر نشده‌اند",
          });
        } else {
          CustomToast({
            type: "error",
            text: "خطا در انجام عملیات",
            des: "لطفا دوباره امتحان کنید",
          });
        }
      },
    });
  };
  return (
    <form className="w-full lg:pl-10" onSubmit={handleSubmit(submitHandler)}>
      <div className="grid  grid-cols-1 md:grid-cols-2 gap-x-16 gap-y-2">
        {inputs.map((input) => (
          <div
            key={input.id}
            className={`child:text-[#696464] dark:child:text-light ${
              input.textarea ? "col-span-1 md:col-span-2" : ""
            }`}
            // className="child:text-[#696464] dark:child:text-light"
          >
            <p className="font text-lg">
              {input.label}{" "}
              {/* {input.required ? <span className="text-error-400">*</span> : ""} */}
            </p>
            {input.textarea ? (
              <textarea
                className="w-full mt-2 bg-[#EFEFEF] dark:bg-dark-500 p-4 text-base-mines rounded-sm outline-none lg:resize-none overflow-hidden"
                placeholder="وارد کنید"
                {...register(input.value)}
                rows={8}
              />
            ) : (
              <input
                className="w-full h-[53px] mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 p-4 text-base-mines rounded-sm"
                type={input.inputType}
                placeholder="وارد کنید"
                {...register(input.value)}
              />
            )}

            {errors[input.value] && (
              <div className="flex items-center gap-2.5 w-full mt-2.5 child:text-[#FF0000] text-xs">
                <svg className="size-5 shrink-0">
                  <use href="#danger"></use>
                </svg>
                <p className="text-wrap whitespace-normal">
                  {errors[input.value]?.message}
                </p>
              </div>
            )}
          </div>
        ))}
      </div>
      <div className="w-full text-right xl:text-left mt-[23px]">
        <Button
          className="hidden sm:inline-block flex-center w-full xl:w-[165px] h-[55px] px-5 bg-secondary-default hover:bg-secondary-default/90 text-base dark:text-light rounded-10"
          type="submit"
          disabled={isPending}
        >
          ارسال
        </Button>
      </div>
    </form>
  );
}
