import React, { useState } from "react";
import { Button } from "@/components/ui/button";
import Image from "next/image";
import { useForm } from "react-hook-form";
import { yupResolver } from "@hookform/resolvers/yup";
import { passwordLoginSchema } from "@/configs/ValidationSchema";
import { useMutation, useQueryClient } from "@tanstack/react-query";
import { loginWithPassword } from "@/services/auth";
import CustomToast from "@/components/modules/CustomToast";
import { setAccessToken } from "@/lib/token";
import { ApiError, ApiResponse } from "@/@types/api";
import { AxiosResponse } from "axios";
type Inputs = {
  phone: string;
  password: string;
};
type LoginPasswordProps = {
  //   step: number;
  setStep: React.Dispatch<React.SetStateAction<number>>;
  setIsLoginModal: React.Dispatch<React.SetStateAction<boolean>>;
};

export default function LoginPassword({
  setStep,
  setIsLoginModal,
}: LoginPasswordProps) {
  const [showPassword, setShowPassword] = useState<boolean>(false);
  const {
    register,
    handleSubmit,
    formState: { errors },
  } = useForm<Inputs>({
    resolver: yupResolver(passwordLoginSchema),
  });
  const queryClient = useQueryClient();
  const { mutate, isPending } = useMutation<
    AxiosResponse<ApiResponse<string>>,
    ApiError,
    Inputs
  >({ mutationFn: loginWithPassword });

  const loginHandler = handleSubmit((data: Inputs) => {
    mutate(data, {
      onSuccess: (res) => {
        if (res.status === 200) {
          CustomToast({ type: "success", text: "با موفقیت وارد شدید" });
          setAccessToken(res.data.data);
          queryClient.invalidateQueries({ queryKey: ["get-me"] });
          setIsLoginModal(false);
        }
      },
      onError: (error) => {
        const statusCode = error?.response?.status;
        if (statusCode === 401) {
          CustomToast({
            type: "error",
            text: "شماره موبایل  یا رمز عبور معتبر نمی باشد",
          });
        } else if (statusCode === 400) {
          CustomToast({
            type: "error",
            text: "کاربری یافت نشد",
          });
        } else {
          CustomToast({
            type: "error",
            text: "خطا در انجام عملیات",
            des: "لطفا دوباره امتحان کنید",
          });
        }
      },
    });
  });
  return (
    <form
      className="fixed top-0 flex items-center justify-center mx-auto w-screen h-screen !z-[100] overflow-x-hidden animate-fadeIn "
      onSubmit={loginHandler}
    >
      <div
        className="absolute  w-full h-full z-40"
        onClick={() => setIsLoginModal(false)}
      ></div>
      <div className="w-[350px] md:w-[400px] p-2.5 mx-auto bg-light rounded-2xl z-50">
        <div className="mx-auto">
          <Image
            src={"/images/logo.svg"}
            alt="عکس لوگو سایت داکوبیت"
            width={100}
            height={60}
            className="w-[130px] mx-auto"
          />
          <h2 className=" my-6 text-neutral-950 font-peyda font-extrabold text-center text-lg ">
            ورود با رمز عبور
          </h2>
          <h3 className="font-medium text-base text-center  text-neutral-950">
            حساب کاربری ندارید؟
            <span
              className=" text-secondary-default hover:text-secondary-default/80 cursor-pointer"
              onClick={() => setStep(3)}
            >
              {" "}
              ثبت نام کنید
            </span>{" "}
          </h3>
        </div>
        <div className="w-full my-4 mx-auto ">
          <div className="flex items-center justify-between w-2/3 h-[50px] p-1.5 mx-auto  border border-neutral-100 rounded-13 overflow-hidden">
            <input
              className="input__ltr w-full h-full px-2.5 bg-inherit text-neutral-600 outline-none"
              type="text"
              placeholder="شماره موبایل"
              {...register("phone")}
            />
            <svg className="size-5 text-neutral-400">
              <use href="#phone"></use>
            </svg>
          </div>
          <div className="w-2/3 mt-2  mx-auto">
            <span className=" text-sm text-error-400">
              {errors?.phone && errors?.phone.message}
            </span>
          </div>
        </div>
        <div className="w-full my-4 mx-auto ">
          <div className="flex items-center justify-between w-2/3 h-[50px] p-1.5 mx-auto  border border-neutral-100 rounded-13 overflow-hidden">
            <input
              className="input__ltr w-full h-full px-2.5 bg-inherit text-neutral-600 outline-none"
              type={showPassword ? "text" : "password"}
              placeholder="رمز عبور"
              {...register("password")}
            />
            {showPassword ? (
              <svg
                className="size-5 text-neutral-400 cursor-pointer"
                onClick={() => setShowPassword(false)}
              >
                <use href="#eye-slash"></use>
              </svg>
            ) : (
              <svg
                className="size-5 text-neutral-400 cursor-pointer"
                onClick={() => setShowPassword(true)}
              >
                <use href="#eye"></use>
              </svg>
            )}
          </div>
          <div className="w-2/3 mt-2  mx-auto">
            <span className=" text-sm text-error-400">
              {errors?.password && errors?.password.message}
            </span>
          </div>
        </div>

        <p
          className=" mt-1.5 mb-4 text-xs text-secondary-default hover:text-secondary-default/80 text-center cursor-pointer"
          onClick={() => setStep(0)}
        >
          ورود با موبایل
        </p>
        {/* <p
          className=" mt-1.5 mb-4 text-xs text-secondary-default hover:text-secondary-default/80 text-center cursor-pointer"
          onClick={() => setStep(0)}
        >
          فراموشی رمز عبور
        </p> */}
        <div className="w-full mx-auto text-center">
          <Button
            className="bg-secondary-default hover:bg-secondary-default/90 text-light border border-light-400 rounded"
            size="lg"
            type="submit"
            disabled={isPending}
          >
            ورود
          </Button>
        </div>
      </div>
    </form>
  );
}
