"use client";
import { ApiError, ApiResponse } from "@/@types/api";
import { OrderCountContext } from "@/app/context/OrderCountProvider";
import { userContext } from "@/app/context/UserProvider";
import CustomToast from "@/components/modules/CustomToast";
import { Button } from "@/components/ui/button";
import { filteredIdInSelectOptions, totalPrice } from "@/lib/utils";
import { getOrders, sendOrder } from "@/services/order";
import { useMutation, useQuery } from "@tanstack/react-query";
import { AxiosResponse } from "axios";
import React, { useContext, useState } from "react";
import { usedDiscount } from "@/services/discount";
import { payment } from "@/services/payment";
import { updateOrder } from "@/services/services/order";

interface ErrorResponse {
  msg: string;
}
interface DiscountResponse {
  discountApplied: number;
  eligibleAmount: number;
  totalAfterDiscount: number;
}

export default function OrderInfo() {
  const contextData = useContext(OrderCountContext);
  const userData = useContext(userContext);
  const [discountText, setDiscountText] = useState<string>("");
  const [isValidDiscount, setIsValidDiscount] = useState<boolean>(false);
  const [discountRes, setDiscountRes] = useState<{
    discountApplied: number;
    eligibleAmount: number;
    totalAfterDiscount: number;
  }>({
    discountApplied: 0,
    eligibleAmount: 0,
    totalAfterDiscount: 0,
  });

  const { data: userOrders } = useQuery({
    queryKey: ["user-orders"],
    queryFn: getOrders,
    select: (res) => res.data,
  });

  const { mutate, isPending } = useMutation<
    AxiosResponse<ApiResponse<string>>,
    ApiError,
    { productid: string; price: number; count: number }[]
  >({ mutationFn: sendOrder });

  const { mutate: updateOrder_mutate, isPending: updateOrder_isPending } =
    useMutation<
      AxiosResponse<ApiResponse<string>>,
      ApiError,
      { productid: string; price: number; count: number }[]
    >({ mutationFn: updateOrder });

  const { mutate: discount_mutate, isPending: disCount_isPending } =
    useMutation<AxiosResponse<ApiResponse<DiscountResponse>>, ApiError, string>(
      {
        mutationFn: usedDiscount,
      }
    );

  const { mutate: payment_mutate } = useMutation<
    AxiosResponse<ApiResponse<string>>,
    ApiError
  >({
    mutationFn: payment,
  });
  // const [isShowPreviewModal, setIsShowPreviewModal] = useState<boolean>(false);

  const paymentHandler = () => {
    payment_mutate(undefined, {
      onSuccess: (res) => {
        CustomToast({
          type: "success",
          text: "در حال ارسال به درگاه پرداخت",
        });
        contextData?.setLocalDate([]);
        contextData?.setOrderCount(0);
        localStorage.removeItem("orders");
        window.location.href = res.data.data;
      },
      onError: (error) => {
        const statusCode = error?.response?.status;
        if (statusCode === 400) {
          CustomToast({
            type: "error",
            text: " درخواست نامعتبر است",
          });
        } else {
          CustomToast({
            type: "error",
            text: "خطا در انجام عملیات",
            des: "لطفا دوباره امتحان کنید",
          });
        }
      },
    });
  };

  const orderHandler = () => {
    if (contextData?.localDate.length === 0) {
      return CustomToast({
        type: "error",
        text: "لطفا ابتدا محصول خود را انتخاب کنید",
      });
    }
    if (userData?.userInfo === null) {
      return CustomToast({
        type: "error",
        text: "لطفا ابتدا وارد حساب کاربری خود شوید",
      });
    }

    const orders = filteredIdInSelectOptions(contextData?.localDate ?? []);
    if (userOrders?.data.length > 0) {
      updateOrder_mutate(orders ?? [], {
        onSuccess: () => {
          CustomToast({
            type: "success",
            text: "سفارش شما با موفقیت ثبت شد",
          });
          paymentHandler();
        },
        onError: (error) => {
          const statusCode = error?.response?.status;
          if (statusCode === 400) {
            CustomToast({
              type: "error",
              text: " دیتا نامعتبر است",
            });
          } else {
            CustomToast({
              type: "error",
              text: "خطا در انجام عملیات",
              des: "لطفا دوباره امتحان کنید",
            });
          }
        },
      });
    } else {
      mutate(orders ?? [], {
        onSuccess: () => {
          CustomToast({
            type: "success",
            text: "سفارش شما با موفقیت ثبت شد",
          });
          paymentHandler();
        },
        onError: (error) => {
          const statusCode = error?.response?.status;
          if (statusCode === 400) {
            CustomToast({
              type: "error",
              text: " دیتا نامعتبر است",
            });
          } else {
            CustomToast({
              type: "error",
              text: "خطا در انجام عملیات",
              des: "لطفا دوباره امتحان کنید",
            });
          }
        },
      });
    }
  };

  const discountHandler = () => {
    if (disCount_isPending) return;
    if (!discountText.trim()) {
      return CustomToast({
        text: "لطفا کد تخفیف خود را وارد کنید",
        type: "error",
      });
    }

    discount_mutate(discountText, {
      onSuccess: (res) => {
        CustomToast({
          type: "success",
          text: "کد تخفیف شما با موفقیت ثبت شد",
        });
        setDiscountRes((prev) => ({
          ...prev,
          discountApplied: res?.data?.data?.discountApplied || 0,
          totalAfterDiscount: res?.data?.data?.totalAfterDiscount || 0,
        }));
        setIsValidDiscount(true);
      },
      onError: (error) => {
        const statusCode = error?.response?.status;
        const errorMsg = (error?.response?.data as ErrorResponse)?.msg;
        if (statusCode === 400 || statusCode === 401) {
          CustomToast({
            type: "error",
            text: errorMsg,
          });
        } else {
          CustomToast({
            type: "error",
            text: "خطا در انجام عملیات",
            des: "لطفا دوباره امتحان کنید",
          });
        }
      },
    });
  };
  return (
    <div className="w-full md:w-[32%] mt-[22px] md:mr-[22px] md:mt-0 p-8 rounded-10 border border-[#F2F2F2] shrink-0">
      <p className="font-peyda text-dark-800 font-bold text-2xl">خلاصه سفارش</p>
      <div className="py-8 text-dark-800 text-base font-medium space-y-6 border-b border-[#E7E7E7]">
        <div className="flex items-center justify-between">
          <p>قیمت محصول</p>
          <p>{totalPrice(contextData?.localDate ?? []).toLocaleString()}</p>
        </div>
        <div className="flex items-center justify-between">
          <p>تخفیف</p>
          <p>% {discountRes?.discountApplied}</p>
        </div>
        {/* <div className="flex items-center justify-between">
          <p>کد تخفیف</p>
          <p>0</p>
        </div> */}
      </div>
      <div className="py-8 text-dark-800 text-base font-medium space-y-6 ">
        <div className="flex items-center justify-between child:font-bold">
          <p>جمع کل</p>
          <p>
            {discountRes?.discountApplied > 0
              ? (
                  totalPrice(contextData?.localDate ?? []) -
                  (totalPrice(contextData?.localDate || []) *
                    discountRes?.discountApplied) /
                    100
                )?.toLocaleString()
              : totalPrice(contextData?.localDate ?? []).toLocaleString()}
          </p>
        </div>
        {/* <div className="flex items-center justify-between">
          <p>زمان دانلود</p>
          <p>0</p>
        </div> */}
        <div className="flex items-center w-full py-2 px-4 rounded-9 border border-[#E7E7E7]">
          <input
            type="text"
            className="w-full"
            placeholder="کد تخفیف"
            value={discountText}
            onChange={(e) => setDiscountText(e.target.value)}
          />
          {isValidDiscount ? (
            <div className={`flex-center size-11 rounded bg-success-400`}>
              <svg className="size-6 text-light">
                <use href="#check-badge"></use>
              </svg>
            </div>
          ) : (
            <div
              className={`flex-center size-11 rounded ${
                discountText ? "bg-secondary-default" : "bg-primary-default"
              }`}
            >
              {discountText ? (
                <span
                  className="text-light cursor-pointer"
                  onClick={discountHandler}
                >
                  ثبت
                </span>
              ) : (
                <svg className="size-6 text-dark-800">
                  <use href="#trailing"></use>
                </svg>
              )}
            </div>
          )}
        </div>
        <Button
          className="hidden sm:inline-block flex-center w-full h-[55px] px-5 mt-4 bg-secondary-default hover:bg-secondary-default/90 text-base dark:text-light rounded-10 select-none"
          type="submit"
          onClick={orderHandler}
          disabled={isPending || updateOrder_isPending}
        >
          ثبت و پرداخت
        </Button>
      </div>
      {/* {isShowPreviewModal && <PreviewModal setIsOpen={setIsShowPreviewModal} />} */}
    </div>
  );
}
