"use client";
import React, { Dispatch, SetStateAction, useEffect, useState } from "react";

import ProductsFilter from "./ProductsFilter";
import ProductsGrid from "./ProductsGrid";
import { useQuery } from "@tanstack/react-query";
import { getProducts } from "@/services/produts";
import LoadingBox from "@/components/modules/LoadingBox";
import { defaultPage } from "@/lib/constants";
import { paginationType } from "@/@types/pagination";
import { FilterType } from "@/@types/filter";
import { useRouter, useSearchParams } from "next/navigation";
import { buildSearchParams } from "@/lib/utils";

type ProductsBoxProps = {
  filter: FilterType;
  setFilter: Dispatch<SetStateAction<FilterType>>;
};

export default function ProductsBox({ filter, setFilter }: ProductsBoxProps) {
  const [paginationSize, setPaginationSize] =
    useState<paginationType>(defaultPage);

  const router = useRouter();
  const searchParams = useSearchParams();

  const searchParamsText = buildSearchParams(filter, {
    page: paginationSize?.page,
  });

  const { data: products, isLoading } = useQuery({
    queryKey: ["products", searchParamsText],
    queryFn: () => getProducts(searchParamsText),
  });

  const buildQueryString = (filters: Record<string, unknown>) => {
    const params = new URLSearchParams();

    Object.entries(filters).forEach(([key, value]) => {
      if (value !== "" && value !== false) {
        params.set(key, String(value));
      }
    });
    Object.entries(paginationSize).forEach(([key, value]) => {
      if (key === "limit" || (key === "page" && value === 1)) return;
      params.set(key, String(value));
    });

    return params.toString();
  };

  useEffect(() => {
    const paramsObj: Record<string, string> = {};

    searchParams.forEach((value, key) => {
      paramsObj[key] = value;
    });

    setFilter((prev) => ({
      ...prev,
      ...paramsObj,
    }));
    setPaginationSize({
      limit: 10,
      page: Number(paramsObj?.page) || 1,
    });
  }, []);

  useEffect(() => {
    const qs = buildQueryString(filter);
    router.replace(`/products?${qs}`);
  }, [filter, paginationSize]);

  return (
    <section className="flex flex-col lg:flex-row items-start gap-3.75 pb-[200px]">
      <ProductsFilter
        filter={filter}
        setFilter={setFilter}
        setPaginationSize={setPaginationSize}
      />
      {isLoading ? (
        <LoadingBox />
      ) : (
        <ProductsGrid
          products={products?.data?.data || []}
          totalPage={products?.data?.totalPages || 1}
          paginationSize={paginationSize}
          setPaginationSize={setPaginationSize}
        />
      )}
    </section>
  );
}
