import { ApiError, ApiResponse } from "@/@types/api";
import CustomToast from "@/components/modules/CustomToast";
import { Button } from "@/components/ui/button";
import { userRegisterSchema } from "@/configs/ValidationSchema";
import { userRegister } from "@/services/auth";
import { yupResolver } from "@hookform/resolvers/yup";
import { useMutation } from "@tanstack/react-query";
import { AxiosResponse } from "axios";
import Image from "next/image";
import React, { useState } from "react";
import { useForm } from "react-hook-form";

type Inputs = {
  phone: string;
  username: string;
  email: string;
  password: string;
  fname: string;
  lname: string;
  refCode: string;
};
type RegisterProps = {
  //   step: number;
  setStep: React.Dispatch<React.SetStateAction<number>>;
  setIsLoginModal: React.Dispatch<React.SetStateAction<boolean>>;
  setPhone: React.Dispatch<React.SetStateAction<string>>;
  setIsRegister: React.Dispatch<React.SetStateAction<boolean>>;
  userInfo: Inputs;
  setUserInfo: React.Dispatch<React.SetStateAction<Inputs>>;
  refCode: string;
};
export default function Register({
  setStep,
  setIsLoginModal,
  setPhone,
  setIsRegister,
  userInfo,
  setUserInfo,
  refCode,
}: RegisterProps) {
  const [showPassword, setShowPassword] = useState<boolean>(false);
  const {
    register,
    handleSubmit,
    formState: { errors },
  } = useForm<Inputs>({
    defaultValues: {
      phone: userInfo.phone ? userInfo.phone : "",
      username: userInfo.username ? userInfo.username : "",
      email: userInfo.email ? userInfo.email : "",
      password: userInfo.password ? userInfo.password : "",
      fname: userInfo.fname ? userInfo.fname : "",
      lname: userInfo.lname ? userInfo.lname : "",
      refCode: refCode || "",
    },
    resolver: yupResolver(userRegisterSchema),
  });

  const { mutate, isPending } = useMutation<
    AxiosResponse<ApiResponse<string>>,
    ApiError,
    Inputs
  >({
    mutationFn: userRegister,
  });
  const registerHandler = handleSubmit((data: Inputs) => {
    mutate(data, {
      onSuccess: (res) => {
        setPhone(data.phone);
        setStep(1);
        setIsRegister(true);
        setUserInfo({ ...data });
        if (res.status === 200) {
          CustomToast({ type: "success", text: "کد تأیید ارسال شد" });
        }
      },
      onError: (error) => {
        const statusCode = error?.response?.status;
        if (statusCode === 402) {
          CustomToast({
            type: "error",
            text: "شماره موبایل یا ایمیل قبلا توی سایت ثبت نام شده است",
          });
        } else {
          CustomToast({
            type: "error",
            text: "خطا در انجام عملیات",
            des: "لطفا دوباره امتحان کنید",
          });
        }
      },
    });
  });

  return (
    <form
      className="fixed top-0 flex items-center justify-center mx-auto w-screen h-screen !z-[100] overflow-x-hidden animate-fadeIn "
      onSubmit={registerHandler}
    >
      <div
        className="absolute w-full h-full z-40"
        onClick={() => setIsLoginModal(false)}
      ></div>
      <div className="w-[350px] md:w-[400px] p-2.5 mx-auto bg-light rounded-2xl z-50">
        <div className="mx-auto">
          <Image
            src={"/images/logo.svg"}
            alt="عکس لوگو سایت داکوبیت"
            width={100}
            height={60}
            className="w-[130px] mx-auto"
          />
          <h2 className=" my-2 text-neutral-950 font-peyda font-extrabold text-center text-lg ">
            عضویت
          </h2>
          <h3 className="font-medium text-base text-center  text-neutral-950">
            قبلا ثبت نام کرده اید؟
            <span
              className=" text-secondary-default hover:text-secondary-default/80 cursor-pointer"
              onClick={() => setStep(0)}
            >
              {" "}
              وارد شوید
            </span>{" "}
          </h3>
        </div>
        <div className="w-full my-1 mx-auto">
          <div className="flex items-center justify-between w-2/3 h-[50px] p-1.5 mx-auto  border border-neutral-100 rounded-13 overflow-hidden">
            <input
              className="input__ltr w-full h-full px-2.5 bg-inherit text-neutral-600 outline-none"
              type="text"
              placeholder="شماره موبایل"
              {...register("phone")}
            />
            <svg className="size-5 text-neutral-400">
              <use href="#phone"></use>
            </svg>
          </div>
          <div className="w-2/3 mt-0.5  mx-auto">
            <span className=" text-xs-mines text-error-400">
              {errors?.phone && errors?.phone.message}
            </span>
          </div>
        </div>
        <div className="w-full my-1 mx-auto">
          <div className="flex items-center justify-between w-2/3 h-[50px] p-1.5 mx-auto  border border-neutral-100 rounded-13 overflow-hidden">
            <input
              className="input__ltr w-full h-full px-2.5 bg-inherit text-neutral-600 outline-none"
              type="text"
              placeholder="نام کاربری"
              {...register("username")}
            />
            <svg className="size-5 text-neutral-400">
              <use href="#at-symbol"></use>
            </svg>
          </div>
          <div className="w-2/3 mt-0.5  mx-auto">
            <span className=" text-xs-mines text-error-400">
              {errors?.username && errors?.username.message}
            </span>
          </div>
        </div>
        <div className="w-full my-1 mx-auto">
          <div className="flex items-center justify-between w-2/3 h-[50px] p-1.5 mx-auto  border border-neutral-100 rounded-13 overflow-hidden">
            <input
              className="input__ltr w-full h-full px-2.5 bg-inherit text-neutral-600 outline-none"
              type="text"
              placeholder="ایمیل"
              {...register("email")}
            />
            <svg className="size-5 text-neutral-400">
              <use href="#envelope"></use>
            </svg>
          </div>
          <div className="w-2/3 mt-0.5  mx-auto">
            <span className=" text-xs-mines text-error-400">
              {errors?.email && errors?.email.message}
            </span>
          </div>
        </div>
        <div className="w-full my-1 mx-auto">
          <div className="flex items-center justify-between w-2/3 h-[50px] p-1.5 mx-auto  border border-neutral-100 rounded-13 overflow-hidden">
            <input
              className=" w-full h-full px-2.5 bg-inherit text-neutral-600 outline-none"
              type="text"
              placeholder="نام"
              {...register("fname")}
            />
            <svg className="size-5 text-neutral-400">
              <use href="#user"></use>
            </svg>
          </div>
          <div className="w-2/3 mt-0.5  mx-auto">
            <span className=" text-xs-mines text-error-400">
              {errors?.fname && errors?.fname.message}
            </span>
          </div>
        </div>
        <div className="w-full my-1 mx-auto">
          <div className="flex items-center justify-between w-2/3 h-[50px] p-1.5 mx-auto  border border-neutral-100 rounded-13 overflow-hidden">
            <input
              className=" w-full h-full px-2.5 bg-inherit text-neutral-600 outline-none"
              type="text"
              placeholder="نام خانوادگی"
              {...register("lname")}
            />
            <svg className="size-5 text-neutral-400">
              <use href="#user"></use>
            </svg>
          </div>
          <div className="w-2/3 mt-0.5  mx-auto">
            <span className=" text-xs-mines text-error-400">
              {errors?.lname && errors?.lname.message}
            </span>
          </div>
        </div>
        <div className="w-full my-1 mx-auto">
          <div className="flex items-center justify-between w-2/3 h-[50px] p-1.5 mx-auto  border border-neutral-100 rounded-13 overflow-hidden">
            <input
              className="input__ltr w-full h-full px-2.5 bg-inherit text-neutral-600 outline-none"
              type={showPassword ? "text" : "password"}
              placeholder="رمز عبور"
              {...register("password")}
            />
            {showPassword ? (
              <svg
                className="size-5 text-neutral-400 cursor-pointer"
                onClick={() => setShowPassword(false)}
              >
                <use href="#eye-slash"></use>
              </svg>
            ) : (
              <svg
                className="size-5 text-neutral-400 cursor-pointer"
                onClick={() => setShowPassword(true)}
              >
                <use href="#eye"></use>
              </svg>
            )}
          </div>
          <div className="w-2/3 mt-0.5  mx-auto">
            <span className=" text-xs-mines text-error-400">
              {errors?.password && errors?.password.message}
            </span>
          </div>
        </div>
        <div className="w-full my-1 mx-auto">
          <div className="flex items-center justify-between w-2/3 h-[50px] p-1.5 mx-auto  border border-neutral-100 rounded-13 overflow-hidden">
            <input
              className="input__ltr w-full h-full px-2.5 bg-inherit text-neutral-600 outline-none"
              type="text"
              placeholder="کد دعوت"
              {...register("refCode")}
            />
            <svg className="size-5 text-neutral-400">
              <use href="#coupon"></use>
            </svg>
          </div>
          <div className="w-2/3 mt-0.5  mx-auto">
            <span className=" text-xs-mines text-error-400">
              {errors?.refCode && errors?.refCode.message}
            </span>
          </div>
        </div>
        <div className="w-full mx-auto text-center">
          <Button
            className="bg-secondary-default hover:bg-secondary-default/90 text-light border border-light-400 rounded"
            size="lg"
            type="submit"
            disabled={isPending}
          >
            ادامه
          </Button>
        </div>
      </div>
    </form>
  );
}
