"use client";
import React, { useState } from "react";
import { Button } from "@/components/ui/button";
import { useMutation, useQueryClient } from "@tanstack/react-query";
import { ApiError, ApiResponse } from "@/@types/api";
import { AxiosResponse } from "axios";
import { DevType } from "@/@types/dev";
import { sendDevInfo } from "@/services/dev";
import { useForm } from "react-hook-form";
import { yupResolver } from "@hookform/resolvers/yup";
import CustomToast from "@/components/modules/CustomToast";
import { devInfoSchema } from "@/configs/ValidationSchema";
import Image from "next/image";

type input = {
  id: number;
  label: string;
  required: boolean;
  inputType: string;
  value: "title" | "bio" | "website" | "email";
  textarea?: boolean;
};

export default function DevInfoForm({ devData }: { devData: DevType }) {
  const [fileError, setFileError] = useState<string | null>(null);
  const [file, setFile] = useState<File[] | null>(null);
  const [isFileError, setIsFileError] = useState<boolean>(false);

  const queryClient = useQueryClient();

  const { mutate, isPending } = useMutation<
    AxiosResponse<ApiResponse<string>>,
    ApiError,
    FormData
  >({ mutationFn: sendDevInfo });

  const {
    register,
    handleSubmit,
    formState: { errors },
    reset,
  } = useForm<DevType>({
    resolver: yupResolver(devInfoSchema),
    defaultValues: {
      title: devData?.title ?? "",
      bio: devData?.bio ?? "",
      website: devData?.website ?? "",
      // phone: devData?.phone ?? "",
      email: devData?.email ?? "",
    },
  });

  const inputs: input[] = [
    {
      id: 1,
      label: " عنوان فروشگاه :",
      required: false,
      inputType: "text",
      value: "title",
    },
    {
      id: 2,
      label: " لینک فروشگاه :",
      required: false,
      inputType: "text",
      value: "website",
    },
    {
      id: 3,
      label: " ایمیل :",
      required: false,
      inputType: "email",
      value: "email",
    },
    {
      id: 4,
      label: "  توضیحات فروشگاه :",
      required: false,
      inputType: "text",
      value: "bio",
      textarea: true,
    },
  ];

  const MAX_FILES = 1;
  const MAX_FILE_SIZE_MB = 1;

  const handleFileChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const files = e.target.files;
    if (!files) return;

    setIsFileError(false);
    setFileError(null);

    const selectedFiles = Array.from(files);

    // محدودیت تعداد
    if (selectedFiles.length + (file?.length || 0) > MAX_FILES) {
      setIsFileError(true);
      setFileError(`حداکثر ${MAX_FILES} فایل می‌توانید انتخاب کنید`);
      return;
    }

    // محدودیت حجم
    const invalidFile = selectedFiles.find(
      (f) => f.size / 1024 / 1024 > MAX_FILE_SIZE_MB
    );
    if (invalidFile) {
      setIsFileError(true);
      setFileError(
        `حجم فایل‌ها نباید بیشتر از ${MAX_FILE_SIZE_MB} مگابایت باشد`
      );
      return;
    }

    // اضافه کردن فایل‌ها به استیت
    setFile((prev) => (prev ? [...prev, ...selectedFiles] : selectedFiles));
    e.target.value = "";
  };
  const submitHandler = (data: DevType) => {
    if (!file || file.length === 0) {
      setIsFileError(true);
      setFileError("لطفا فایل خود را انتخاب کنید");
      return;
    }

    const formData = new FormData();

    Object.entries(data).forEach(([key, value]) => {
      formData.append(key, String(value));
    });

    file.forEach((f) => {
      formData.append("img", f);
    });

    mutate(formData, {
      onSuccess: (res) => {
        if (res.status === 201) {
          CustomToast({
            type: "success",
            text: "مشخصات با موفقیت ثبت شد",
            des: "درخواست شما در حال برسی توسط کارشناسان سایت می باشد",
          });
          reset({
            title: "",
            bio: "",
            website: "",
            // phone: "",
            email: "",
          });
          queryClient.invalidateQueries({ queryKey: ["devInfo"] });
        }
      },
      onError: (error) => {
        const statusCode = error?.response?.status;
        if (statusCode === 401) {
          CustomToast({
            type: "error",
            text: "عدم دسترسی به ارسال دیتا",
          });
        } else if (statusCode === 400) {
          CustomToast({
            type: "error",
            text: "درخواست قبلاً ثبت شده یا ورودی نامعتبر",
          });
        } else {
          CustomToast({
            type: "error",
            text: "خطا در انجام عملیات",
            des: "لطفا دوباره امتحان کنید",
          });
        }
      },
    });
  };

  return (
    <section className="w-full mt-6 py-6 px-[29px] bg-light dark:bg-[#393939]">
      <form onSubmit={handleSubmit(submitHandler)}>
        <div>
          <p className="font-bold text-xl text-[#001A46] dark:text-light">
            مشخصات توسعه دهنده
          </p>
          <div className="flex items-center mt-4">
            <span className="w-[10%] h-0.5 bg-secondary-default dark:bg-primary-200"></span>
            <span className="w-[90%] h-0.5 bg-dark/[7%] dark:bg-[#4A4A4A]"></span>
          </div>
        </div>
        <div className="grid  grid-cols-1 md:grid-cols-2 xl:grid-cols-3 gap-x-7 xl:gap-x-10 gap-y-7 xl:gap-y-[32px] mt-10">
          {inputs.map((input) => (
            <div
              key={input.id}
              className={`child:text-[#696464] dark:child:text-light ${
                input.textarea ? "col-span-1 md:col-span-2 xl:col-span-3" : ""
              }`}
            >
              <p className="font text-lg">
                {input.label}{" "}
                {input.required ? (
                  <span className="text-error-400">*</span>
                ) : (
                  ""
                )}
              </p>
              {input.textarea ? (
                <textarea
                  className="w-full h-auto lg:h-[100px] mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 p-4 text-base-mines rounded-sm outline-none lg:resize-none"
                  placeholder="وارد کنید"
                  {...register(input.value)}
                  rows={4}
                />
              ) : (
                <input
                  className="w-full h-[53px] mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 p-4 text-base-mines rounded-sm"
                  type={input.inputType}
                  placeholder="وارد کنید"
                  {...register(input.value)}
                />
              )}

              {errors[input.value] && (
                <div className="flex items-center gap-2.5 w-full mt-2.5 child:text-[#FF0000] text-xs">
                  <svg className="size-5 shrink-0">
                    <use href="#danger"></use>
                  </svg>
                  <p className="text-wrap whitespace-normal">
                    {errors[input.value]?.message}
                  </p>
                </div>
              )}
            </div>
          ))}
          {/* uploader */}
          <div>
            <p className="font text-lg child:text-[#696464] dark:child:text-light">
              <span className="font text-lg">عکس بنر فروشگاه :</span>
              <span className="font text-xs">(ابعاد عکس: 400 * 1200 )</span>
            </p>
            <label className="flex items-center justify-between w-full sm:w-[180px] py-4 px-6 mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 rounded-md border border-dashed border-[#DDDDDD] dark:border-[#9B9B9B]/50 cursor-pointer">
              <span className="text-[#9B9B9B]">آپلود پیوست</span>
              <input
                className="opacity-0 w-0 h-0"
                type="file"
                multiple
                onChange={handleFileChange}
              />
              <svg className="w-6 h-6 text-[#9B9B9B]">
                <use href="#file"></use>
              </svg>
            </label>

            {isFileError && fileError && (
              <div className="flex items-center gap-2.5 w-full mt-2.5 child:text-[#FF0000] text-xs">
                <svg className="size-5 shrink-0">
                  <use href="#danger"></use>
                </svg>
                <p className="text-wrap whitespace-normal">{fileError}</p>
              </div>
            )}
          </div>
          <div className="col-span-1 md:col-span-2 xl:col-span-3 flex flex-wrap gap-2 my-2.5">
            {file?.map((f, index) => (
              <div
                key={index}
                className="relative w-full h-auto border rounded overflow-hidden"
              >
                <Image
                  src={URL.createObjectURL(f)}
                  alt={f.name}
                  width={1200}
                  height={400}
                  className="w-full h-full object-cover"
                />
                <Button
                  className="absolute top-1 right-1 bg-[#FF0000] hover:bg-white hover:text-[#FF0000] text-white w-5 h-5 p-0 m-0 flex items-center justify-center rounded-full"
                  onClick={() =>
                    setFile((prev) =>
                      prev ? prev.filter((_, i) => i !== index) : null
                    )
                  }
                >
                  <svg>
                    <use href="#x-mark"></use>
                  </svg>
                </Button>
              </div>
            ))}
          </div>

          <div className="w-full mt-auto">
            <Button
              className="inline-block flex-center w-full h-[53px] px-5 bg-secondary-default hover:bg-secondary-default/90 text-base dark:text-light rounded"
              type="submit"
              disabled={isPending}
            >
              ذخیره
            </Button>
          </div>
        </div>
      </form>
    </section>
  );
}
