"use client";

import React, { useState } from "react";

import { Button } from "@/components/ui/button";
import { devDiscountSchema } from "@/configs/ValidationSchema";
import { yupResolver } from "@hookform/resolvers/yup";
import { Controller, useForm } from "react-hook-form";
import DatePicker, { DateObject } from "react-multi-date-picker";
import persian from "react-date-object/calendars/persian";
import persian_fa from "react-date-object/locales/persian_fa";
import gregorian from "react-date-object/calendars/gregorian";
import { useMutation, useQueryClient } from "@tanstack/react-query";
import { AxiosResponse } from "axios";
import { ApiError, ApiResponse } from "@/@types/api";
import { senDevDiscount } from "@/services/dev";
import CustomToast from "@/components/modules/CustomToast";
import FileComboBox from "./FileComboBox";

type InputType = {
  id: number;
  label: string;
  inputType: string;
  value: "code" | "amount" | "usageLimit";
};

type Inputs = {
  code: string;
  amount: string;
  usageLimit: string;
  expiry: string;
};

export default function AddDiscountForm() {
  const queryClient = useQueryClient();
  const [expiryDate, setExpiryDate] = useState<DateObject | null>(null);
  const {
    register,
    handleSubmit,
    formState: { errors },
    control,
    reset,
  } = useForm<Inputs>({
    resolver: yupResolver(devDiscountSchema),
  });
  const { mutate, isPending } = useMutation<
    AxiosResponse<ApiResponse<string>>,
    ApiError,
    {
      code: string;
      amount: number;
      usageLimit: number;
      expiry: string;
      fileid: string;
      usedCount: number;
    }
  >({ mutationFn: senDevDiscount });
  const inputs: InputType[] = [
    {
      id: 1,
      label: "کد تخفیف :",
      inputType: "text",
      value: "code",
    },
    {
      id: 2,
      label: "میزان تخفیف :",
      inputType: "text",
      value: "amount",
    },
    {
      id: 3,
      label: "تعداد کل استفاده :",
      inputType: "text",
      value: "usageLimit",
    },
  ];

  const submitHandler = (data: {
    code: string;
    amount: string;
    usageLimit: string;
    expiry: string;
  }) => {
    mutate(
      {
        ...data,
        fileid: "64f6a1c9a5b3e123456789ab",
        usedCount: 0,
        amount: +data.amount,
        usageLimit: +data.usageLimit,
      },
      {
        onSuccess: (res) => {
          if (res.status === 200) {
            CustomToast({
              type: "success",
              text: "کد تخفیف با موفقیت ایجاد شد",
            });
            reset({
              code: "",
              amount: "",
              usageLimit: "",
              expiry: "",
            });
            setExpiryDate(null);
            queryClient.invalidateQueries({ queryKey: ["devDiscount"] });
          }
        },
        onError: (error) => {
          const statusCode = error?.response?.status;
          if (statusCode === 401) {
            CustomToast({
              type: "error",
              text: "عدم دسترسی به ارسال دیتا",
            });
          } else if (statusCode === 400) {
            CustomToast({
              type: "error",
              text: "درخواست قبلاً ثبت شده یا ورودی نامعتبر",
            });
          } else {
            CustomToast({
              type: "error",
              text: "خطا در انجام عملیات",
              des: "لطفا دوباره امتحان کنید",
            });
          }
        },
      }
    );
  };

  return (
    <form onSubmit={handleSubmit(submitHandler)}>
      <div className="grid  grid-cols-1 md:grid-cols-2 xl:grid-cols-3 gap-x-7 xl:gap-x-10 gap-y-7 xl:gap-y-[32px] mt-10">
        {inputs.map((input) => (
          <div
            key={input.id}
            className="child:text-[#696464] dark:child:text-light"
          >
            <p className="font text-lg">{input.label} </p>

            <input
              className="w-full h-[53px] mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 p-4 text-base-mines rounded-sm"
              type={input.inputType}
              placeholder="وارد کنید"
              {...register(input.value)}
            />

            {errors[input.value] && (
              <div className="flex items-center gap-2.5 w-full mt-2.5 child:text-[#FF0000] text-xs">
                <svg className="size-5 shrink-0">
                  <use href="#danger"></use>
                </svg>
                <p className="text-wrap whitespace-normal">
                  {errors[input.value]?.message}
                </p>
              </div>
            )}
          </div>
        ))}
        <div className="child:text-[#696464] dark:child:text-light">
          <p className="font text-lg">تاریخ انقضا : </p>

          <Controller
            control={control}
            name="expiry"
            rules={{ required: true }}
            render={({ field: { onChange } }) => (
              <div className="w-full h-full">
                <div className="flex items-center justify-between w-full h-[53px] mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 p-4 pl-10 text-base-mines rounded-sm outline-none">
                  <DatePicker
                    value={expiryDate}
                    format="YYYY/MM/DD"
                    style={{
                      background: "inherit",
                      width: "150px",
                      height: "53px",
                      padding: "16px",
                    }}
                    className="!w-full"
                    onChange={(date) => {
                      if (!date) {
                        setExpiryDate(null);
                        onChange("");
                        return;
                      }

                      setExpiryDate(date as DateObject);
                      const g = (date as DateObject).convert(gregorian);

                      const year = g.year;
                      const month = g.month.number;
                      const day = g.day;

                      const utcMs = Date.UTC(year, month - 1, day, 0, 0, 0, 0);
                      const isoUtc = new Date(utcMs)
                        .toISOString()
                        .replace("Z", "+00:00");

                      onChange(isoUtc);
                    }}
                    placeholder="وارد کنید"
                    inputClass="datePicker_style"
                    calendar={persian}
                    locale={persian_fa}
                  />
                  <svg className="w-5 h-5 text-tertiary-40 shrink-0">
                    <use href="#date"></use>
                  </svg>
                </div>
                {errors.expiry && (
                  <div className="flex items-center gap-2.5 w-full mt-2.5 child:text-[#FF0000] text-xs">
                    <svg className="size-5 shrink-0">
                      <use href="#danger"></use>
                    </svg>
                    <p className="text-wrap whitespace-normal">
                      {errors.expiry?.message}
                    </p>
                  </div>
                )}
              </div>
            )}
          />
        </div>
        <FileComboBox />
        <div className="w-full my-auto">
          <Button
            className="inline-block flex-center w-full h-[53px] px-5 lg:mt-9 bg-secondary-default hover:bg-secondary-default/90 text-base dark:text-light rounded"
            type="submit"
            disabled={isPending}
          >
            ذخیره
          </Button>
        </div>
      </div>
    </form>
  );
}
