import React, { Dispatch, SetStateAction } from "react";

import { ApiError, ApiResponse } from "@/@types/api";
import CustomToast from "@/components/modules/CustomToast";
import { Button } from "@/components/ui/button";
import { setAccessToken } from "@/lib/token";
import { deleteDevDiscount } from "@/services/dev";
import { useMutation, useQueryClient } from "@tanstack/react-query";
import { AxiosResponse } from "axios";

type DeleteModalProps = {
  setIsOpenDeleteModal: Dispatch<SetStateAction<boolean>>;
  discountId: string;
};

export default function DeleteModal({
  setIsOpenDeleteModal,
  discountId,
}: DeleteModalProps) {
  const queryClient = useQueryClient();
  const { mutate, isPending } = useMutation<
    AxiosResponse<ApiResponse<string>>,
    ApiError,
    string
  >({ mutationFn: deleteDevDiscount });

  const deleteHandler = () => {
    mutate(discountId, {
      onSuccess: (res) => {
        if (res.status === 200) {
          CustomToast({ type: "success", text: "کد تخفیف با موفقیت حذف شد" });
          setAccessToken(res.data.data);
          queryClient.invalidateQueries({ queryKey: ["devDiscount"] });
          setIsOpenDeleteModal(false);
        }
      },
      onError: (error) => {
        const statusCode = error?.response?.status;
        if (statusCode === 401) {
          CustomToast({
            type: "error",
            text: "عدم دسترسی به ارسال دیتا",
          });
        } else if (statusCode === 404) {
          CustomToast({
            type: "error",
            text: "کد تخفیف یافت نشد",
          });
        } else {
          CustomToast({
            type: "error",
            text: "خطا در انجام عملیات",
            des: "لطفا دوباره امتحان کنید",
          });
        }
        setIsOpenDeleteModal(false);
      },
    });
  };
  return (
    <div className="fixed inset-0 flex items-center justify-center mx-auto w-screen h-screen overflow-hidden animate-fadeIn z-[100]">
      <div className="w-[350px] md:w-[400px] p-3 bg-light dark:bg-dark-400 rounded-10 z-50">
        <div>
          <p className="p-4 font-peyda font-bold text-lg  text-center">
            آیا از حذف کد تخفیف مطمئن هستید؟
          </p>
        </div>
        <div className="flex-center gap-2.5 w-full py-2">
          <Button
            className="inline-block flex-center h-10 px-5 bg-secondary-default hover:bg-secondary-default/90 text-base text-light hover:text-light rounded"
            variant="outline"
            onClick={() => {
              setIsOpenDeleteModal(false);
            }}
          >
            لغو
          </Button>
          <Button
            className="inline-block flex-center h-10 px-5 bg-red-600 hover:bg-red-500 text-base text-light rounded"
            onClick={deleteHandler}
            disabled={isPending}
          >
            تایید
          </Button>
        </div>
      </div>

      <div
        className="absolute w-full h-full z-40"
        onClick={() => {
          setIsOpenDeleteModal(false);
        }}
      ></div>
    </div>
  );
}
