import { ApiError, ApiResponse } from "@/@types/api";
import { bankDataType } from "@/@types/Bank";
import CustomToast from "@/components/modules/CustomToast";
import { Button } from "@/components/ui/button";
import { sendWithdrawal } from "@/services/withdrawal";
import { useMutation, useQueryClient } from "@tanstack/react-query";
import { AxiosResponse } from "axios";
import Image from "next/image";
import React, { Dispatch, SetStateAction, useEffect, useState } from "react";

type WithdrawRequestModalProps = {
  setIsOpenWithdrawRequestModal: Dispatch<SetStateAction<boolean>>;
  banks: bankDataType[];
};
type input = {
  id: number;
  label: string;
  required: boolean;
  inputType: string;
  value: "amount";
  textarea?: boolean;
  ltr?: boolean;
  leftDir?: boolean;
  placeholder: string;
};

export default function WithdrawRequestModal({
  setIsOpenWithdrawRequestModal,
  banks,
}: WithdrawRequestModalProps) {
  const [userBankInfo, setUserBankInfo] = useState({
    bankid: "",
    bankIcon: "",
    amount: "",
    cardNumber: "",
  });

  const [bankError, setBankError] = useState("");
  const [amountError, setAmountError] = useState("");
  const [isShowOptions, setIShowOptions] = useState(false);
  const inputs: input[] = [
    {
      id: 1,
      label: "مبلغ برداشت :",
      required: false,
      inputType: "text",
      value: "amount",
      placeholder: "مبلغ خود را وارد کنید",
    },
  ];
  const queryClient = useQueryClient();
  const { mutate } = useMutation<
    AxiosResponse<ApiResponse<string>>,
    ApiError,
    { bankid: string; amount: number }
  >({
    mutationFn: sendWithdrawal,
  });
  useEffect(() => {
    if (userBankInfo.bankid) {
      setBankError("");
    }
    if (userBankInfo.amount) {
      setAmountError("");
    }
  }, [userBankInfo]);

  const clickHandler = (id: string, cardNumber: string, bankName: string) => {
    setUserBankInfo((prev) => ({
      ...prev,
      bankid: id,
      cardNumber,
      bankIcon: bankName,
    }));
    setIShowOptions(true);
  };
  const submitHandler = () => {
    if (!userBankInfo?.bankIcon.trim() || !userBankInfo?.amount.trim()) {
      if (!userBankInfo?.bankIcon) {
        setBankError("لطفا شماره کارت خود را انتخاب کنید");
      }
      if (!userBankInfo?.amount) {
        setAmountError("لطفا مبلغ خود را وارد کنید");
      }
      return;
    }
    mutate(
      { bankid: userBankInfo?.bankid, amount: +userBankInfo?.amount },
      {
        onSuccess: (res) => {
          if (res.status === 200) {
            CustomToast({
              type: "success",
              text: "درخواست شما با موفقیت ثبت شد",
            });
          }
          setUserBankInfo({
            bankid: "",
            bankIcon: "",
            amount: "",
            cardNumber: "",
          });
          setIsOpenWithdrawRequestModal(false);
          queryClient.invalidateQueries({ queryKey: ["user-withdrawal"] });
        },
        onError: (error) => {
          const statusCode = error?.response?.status;
          if (statusCode === 400) {
            CustomToast({
              type: "error",
              text: "ورودی معتبر نمی باشد",
            });
          } else if (statusCode === 401) {
            CustomToast({
              type: "error",
              text: "عدم دسترسی به ارسال دیتا",
            });
          } else {
            CustomToast({
              type: "error",
              text: "خطا در انجام عملیات",
              des: "لطفا دوباره امتحان کنید",
            });
          }
        },
      }
    );
  };
  return (
    <div className="fixed inset-0 flex items-center justify-center mx-auto w-screen h-screen overflow-hidden animate-fadeIn z-50">
      <div className="w-[350px] md:w-[600px] p-3 bg-light dark:bg-dark-400 rounded-10 z-50">
        <div>
          <div className="flex items-center justify-between">
            <div className="flex items-center gap-1.5">
              <p className="font-bold text-base text-[#001A46] dark:text-light">
                درخواست برداشت
              </p>
            </div>
            <span
              className="inline-flex items-center justify-center size-7 rounded-sm bg-error-400 hover:bg-error-400/90 text-light duration-150 cursor-pointer"
              onClick={() => setIsOpenWithdrawRequestModal(false)}
            >
              X
            </span>
          </div>
          <div className="flex items-center mt-4">
            <span className="w-[20%] h-0.5 bg-secondary-default dark:bg-primary-200"></span>
            <span className="w-[80%] h-0.5 bg-dark/[7%] dark:bg-[#4A4A4A]"></span>
          </div>
        </div>
        <div className="w-full">
          <div className="grid  grid-cols-1 xl:grid-cols-2 gap-x-5 gap-y-7  mt-10">
            <div className={" child:text-[#696464] dark:child:text-light"}>
              <p className="font text-lg">بانک:</p>
              <div
                className="relative w-full h-[53px] mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 py-1 px-4 text-base-mines rounded-sm cursor-pointer"
                onClick={() => setIShowOptions((prev) => !prev)}
              >
                {userBankInfo?.bankid ? (
                  <div className=" flex items-center gap-2">
                    <div>
                      <Image
                        src={`/images/banks/${userBankInfo.bankIcon}.svg`}
                        width={48}
                        height={48}
                        alt="عکس بانک ملی"
                        className="w-12 h-12"
                      />
                    </div>
                    <p>{userBankInfo?.cardNumber}</p>
                  </div>
                ) : (
                  <p className="text-[#696464]/60 dark:text-light-500/65 py-3">
                    لطفا شماره کارت خود را انتخاب کنید
                  </p>
                )}
                <ul
                  className={`absolute top-16 right-0 w-full bg-white rounded-md shadow-2xl border border-muted-foreground/20 hover:child:bg-muted-foreground/20 divide-y divide-muted-foreground/20 ${
                    isShowOptions ? "inline-block" : "hidden"
                  }`}
                >
                  {banks.map((bank: bankDataType) => (
                    <li
                      key={bank?._id}
                      className="flex items-center gap-2 p-2"
                      onClick={() =>
                        clickHandler(
                          bank?._id,
                          bank?.cardNumber,
                          bank?.bankName
                        )
                      }
                    >
                      <div>
                        <Image
                          src={`/images/banks/${bank?.bankName}.svg`}
                          width={48}
                          height={48}
                          alt="عکس بانک ملی"
                          className="w-12 h-12"
                        />
                      </div>
                      <p>{bank?.cardNumber}</p>
                    </li>
                  ))}
                </ul>
              </div>
              {bankError && (
                <div className="flex items-center gap-2.5 w-full mt-2.5 child:text-[#FF0000] text-xs">
                  <svg className="size-5 shrink-0">
                    <use href="#danger"></use>
                  </svg>
                  <p className="text-wrap whitespace-normal">{bankError}</p>
                </div>
              )}
            </div>
            {inputs.map((input) => (
              <div
                key={input.id}
                className={"child:text-[#696464] dark:child:text-light"}
              >
                <p className="font text-lg">{input.label}</p>

                <input
                  className={`w-full h-[53px] mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 p-4 text-base-mines rounded-sm ${
                    input.ltr ? "input__ltr" : ""
                  } ${input.leftDir ? "input__left" : ""}`}
                  name={input?.label}
                  type="number"
                  placeholder={input?.placeholder}
                  value={userBankInfo?.amount}
                  onChange={(e) => {
                    if (isNaN(+e.target.value)) return;
                    setUserBankInfo((prev) => ({
                      ...prev,
                      amount: e.target.value,
                    }));
                  }}
                />
                {amountError && (
                  <div className="flex items-center gap-2.5 w-full mt-2.5 child:text-[#FF0000] text-xs">
                    <svg className="size-5 shrink-0">
                      <use href="#danger"></use>
                    </svg>
                    <p className="text-wrap whitespace-normal">{amountError}</p>
                  </div>
                )}
              </div>
            ))}
          </div>
          <div className="flex items-center justify-center gap-2.5 w-full mt-[32px]">
            <Button
              className="inline-block flex-center w-1/4 h-[53px] px-5  bg-secondary-default hover:bg-secondary-default/90 text-base dark:text-light rounded"
              //   disabled={isPending}
              onClick={submitHandler}
            >
              درخواست
            </Button>
            <Button
              className="inline-block flex-center w-1/4 h-[53px] px-5  bg-error-300 hover:bg-error-300/90 text-base text-light rounded"
              //   type="submit"
              //   disabled={isPending}
              onClick={() => setIsOpenWithdrawRequestModal(false)}
            >
              لغو
            </Button>
          </div>
        </div>
      </div>

      <div
        className="absolute  w-full h-full z-40"
        onClick={() => {
          setIsOpenWithdrawRequestModal(false);
        }}
      ></div>
    </div>
  );
}
