"use client";
import { ApiError, ApiResponse } from "@/@types/api";
import CustomToast from "@/components/modules/CustomToast";
import { Button } from "@/components/ui/button";
import { userAddressSchema } from "@/configs/ValidationSchema";
import { sendUserAddress } from "@/services/auth";
import { yupResolver } from "@hookform/resolvers/yup";
import { useMutation, useQueryClient } from "@tanstack/react-query";
import { AxiosResponse } from "axios";
import React from "react";
import { useForm } from "react-hook-form";
type UserAddersType = {
  website: string;
  phone: string;
  homephone: string;
  postalcode: string;
  city: string;
  province: string;
  address: string;
};
type input = {
  id: number;
  label: string;
  required: boolean;
  inputType: string;
  value:
    | "website"
    | "homephone"
    | "postalcode"
    | "city"
    | "province"
    | "phone"
    | "address";
  textarea?: boolean;
};

type UserAddressFormProps = {
  address: string;
  city: string;
  homephone: string;
  phone: string;
  postalcode: string;
  province: string;
  website: string;
};
export default function UserAddressForm({
  userAddressInfo,
}: {
  userAddressInfo: UserAddressFormProps | null;
}) {
  const queryClient = useQueryClient();
  const { mutate, isPending } = useMutation<
    AxiosResponse<ApiResponse<string>>,
    ApiError,
    UserAddersType
  >({ mutationFn: sendUserAddress });
  const {
    register,
    handleSubmit,
    formState: { errors },
  } = useForm<UserAddersType>({
    resolver: yupResolver(userAddressSchema),
    defaultValues: {
      website: userAddressInfo?.website ?? "",
      phone: userAddressInfo?.phone ?? "",
      homephone: userAddressInfo?.homephone ?? "",
      postalcode: userAddressInfo?.postalcode ?? "",
      city: userAddressInfo?.city ?? "",
      province: userAddressInfo?.province ?? "",
      address: userAddressInfo?.address ?? "",
    },
  });

  const inputs: input[] = [
    {
      id: 1,
      label: " وبسایت :",
      required: false,
      inputType: "text",
      value: "website",
    },
    {
      id: 2,
      label: " شماره موبایل :",
      required: false,
      inputType: "text",
      value: "phone",
    },
    {
      id: 3,
      label: "تلفن ثابت :",
      required: false,
      inputType: "text",
      value: "homephone",
    },
    {
      id: 4,
      label: " کد پستی :",
      required: false,
      inputType: "text",
      value: "postalcode",
    },
    {
      id: 5,
      label: " شهر :",
      required: false,
      inputType: "text",
      value: "city",
    },
    {
      id: 6,
      label: "استان :",
      required: false,
      inputType: "text",
      value: "province",
    },
    {
      id: 7,
      label: " آدرس :",
      required: false,
      inputType: "text",
      value: "address",
      textarea: true,
    },
  ];
  const submitHandler = (data: UserAddersType) => {
    mutate(data, {
      onSuccess: (res) => {
        if (res.status === 200) {
          CustomToast({
            type: "success",
            text: "مشخصات با موفقیت ثبت شد",
          });
        }

        queryClient.invalidateQueries({ queryKey: ["get-me"] });
      },
      onError: (error) => {
        const statusCode = error?.response?.status;
        if (statusCode === 401) {
          CustomToast({
            type: "error",
            text: "عدم دسترسی به ارسال دیتا",
          });
        } else if (statusCode === 400) {
          CustomToast({
            type: "error",
            text: "مشخصات ارسال شده معتبر نمی باشد",
            des: "لطفا دوباره امتحان کنید",
          });
        } else {
          CustomToast({
            type: "error",
            text: "خطا در انجام عملیات",
            des: "لطفا دوباره امتحان کنید",
          });
        }
      },
    });
  };
  return (
    <section className="w-full mt-6 py-6 px-[29px] bg-light dark:bg-[#393939]">
      <form onSubmit={handleSubmit(submitHandler)}>
        <div>
          <p className="font-bold text-xl text-[#001A46] dark:text-light">
            آدرس و تماس
          </p>
          <div className="flex items-center mt-4">
            <span className="w-[10%] h-0.5 bg-secondary-default dark:bg-primary-200"></span>
            <span className="w-[90%] h-0.5 bg-dark/[7%] dark:bg-[#4A4A4A]"></span>
          </div>
        </div>
        <div className="grid  grid-cols-1 md:grid-cols-2 xl:grid-cols-3 gap-x-7 xl:gap-x-10 gap-y-7 xl:gap-y-[32px] mt-10">
          {inputs.map((input) => (
            <div
              key={input.id}
              className={`child:text-[#696464] dark:child:text-light ${
                input.textarea ? "col-span-1 md:col-span-2 xl:col-span-3" : ""
              }`}
            >
              <p className="font text-lg">
                {input.label}{" "}
                {input.required ? (
                  <span className="text-error-400">*</span>
                ) : (
                  ""
                )}
              </p>
              {input.textarea ? (
                <textarea
                  className="w-full h-auto lg:h-[53px] mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 p-4 text-base-mines rounded-sm outline-none lg:resize-none overflow-hidden"
                  placeholder="وارد کنید"
                  {...register(input.value)}
                  rows={4}
                />
              ) : (
                <input
                  className="w-full h-[53px] mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 p-4 text-base-mines rounded-sm"
                  type={input.inputType}
                  placeholder="وارد کنید"
                  {...register(input.value)}
                />
              )}

              {errors[input.value] && (
                <div className="flex items-center gap-2.5 w-full mt-2.5 child:text-[#FF0000] text-xs">
                  <svg className="size-5 shrink-0">
                    <use href="#danger"></use>
                  </svg>
                  <p className="text-wrap whitespace-normal">
                    {errors[input.value]?.message}
                  </p>
                </div>
              )}
            </div>
          ))}
          <div className="w-full text-right xl:text-left mt-auto xl:col-span-3">
            <Button
              className="hidden sm:inline-block flex-center w-full xl:w-1/3 h-[53px] px-5 bg-secondary-default hover:bg-secondary-default/90 text-base dark:text-light rounded"
              type="submit"
              disabled={isPending}
            >
              ذخیره
            </Button>
          </div>
        </div>
      </form>
    </section>
  );
}
