"use client";

import React, { useEffect, useState } from "react";
import { Controller, useForm } from "react-hook-form";
import DatePicker, { DateObject } from "react-multi-date-picker";
import persian from "react-date-object/calendars/persian";
import persian_fa from "react-date-object/locales/persian_fa";
import gregorian from "react-date-object/calendars/gregorian";
import { yupResolver } from "@hookform/resolvers/yup";
import { userInfoSchema } from "@/configs/ValidationSchema";
import { Button } from "@/components/ui/button";
import { UserInfoType } from "@/@types/user";
import { sendUser } from "@/services/auth";
import { useMutation, useQueryClient } from "@tanstack/react-query";
import { ApiError, ApiResponse } from "@/@types/api";
import { AxiosResponse } from "axios";
import CustomToast from "@/components/modules/CustomToast";
import { toEnglishDigits } from "@/lib/utils";
type input = {
  id: number;
  label: string;
  required: boolean;
  inputType: string;
  value: "username" | "fname" | "lname" | "codemeli" | "email";
};

export default function UserInfoForm({
  userInfo,
}: {
  userInfo: UserInfoType | null;
}) {
  const queryClient = useQueryClient();
  const { mutate, isPending: sendUserInfo_isPending } = useMutation<
    AxiosResponse<ApiResponse<string>>,
    ApiError,
    {
      username?: string;
      email: string;
      codemeli: string;
      fname: string;
      lname: string;
      gender: "مرد" | "زن";
      datebirth: string;
      avatar: string;
    }
  >({ mutationFn: sendUser });
  const [dateValue, setDateValue] = useState<DateObject | string>("");
  const {
    register,
    handleSubmit,
    formState: { errors, dirtyFields },
    control,
    reset,
  } = useForm<UserInfoType>({
    resolver: yupResolver(userInfoSchema),
    defaultValues: {
      username: "",
      email: "",
      codemeli: "",
      fname: "",
      lname: "",
      gender: "مرد",
      datebirth: "",
      avatar: "",
    },
  });
  const inputs: input[] = [
    {
      id: 1,
      label: "نام کاربری :",
      required: false,
      inputType: "text",
      value: "username",
    },
    {
      id: 2,
      label: "نام :",
      required: true,
      inputType: "text",
      value: "fname",
    },
    {
      id: 3,
      label: "نام خانوادگی :",
      required: true,
      inputType: "text",
      value: "lname",
    },
    {
      id: 4,
      label: "کد ملی :",
      required: false,
      inputType: "text",
      value: "codemeli",
    },
    {
      id: 5,
      label: "ایمیل :",
      required: true,
      inputType: "email",
      value: "email",
    },
  ];
  useEffect(() => {
    if (userInfo) {
      reset({
        username: userInfo?.username ?? "",
        email: userInfo?.email ?? "",
        codemeli: userInfo?.codemeli ?? "",
        fname: userInfo?.fname ?? "",
        lname: userInfo?.lname ?? "",
        gender: userInfo?.gender ?? "مرد",
        datebirth: userInfo?.datebirth ?? "",
        avatar: userInfo?.avatar ?? "",
      });
      if (userInfo.datebirth) {
        const converDate = toEnglishDigits(userInfo.datebirth);
        const persianDate = new DateObject({
          date: converDate,
          calendar: gregorian,
          format: "YYYY-MM-DD",
        }).convert(persian);
        setDateValue(persianDate);
      } else {
        setDateValue("");
      }
    }
  }, [userInfo, reset]);
  const formSubmit = (data: UserInfoType) => {
    const { username, ...rest } = data;
    const finalData: {
      username?: string;
      email: string;
      codemeli: string;
      fname: string;
      lname: string;
      gender: "مرد" | "زن";
      datebirth: string;
      avatar: string;
    } = {
      ...rest,
      ...(dirtyFields?.username && { username }),
    };
    mutate(finalData, {
      onSuccess: (res) => {
        if (res.status === 200) {
          CustomToast({
            type: "success",
            text: "تغییرات با موفقیت ثبت شد",
          });
          reset(finalData);
        }

        queryClient.invalidateQueries({ queryKey: ["get-me"] });
      },
      onError: (error) => {
        const statusCode = error?.response?.status;
        const errorMsg =
          (error?.response?.data as { msg?: string })?.msg ?? "خطا رخ داده است";
        if (statusCode === 401) {
          CustomToast({
            type: "error",
            text: errorMsg,
          });
        } else if (statusCode === 400) {
          CustomToast({
            type: "error",
            text: "مشخصات ارسال شده معتبر نمی باشد",
            des: "لطفا دوباره امتحان کنید",
          });
        } else {
          CustomToast({
            type: "error",
            text: "خطا در انجام عملیات",
            des: "لطفا دوباره امتحان کنید",
          });
        }
      },
    });
  };
  return (
    <form className="w-full" onSubmit={handleSubmit(formSubmit)}>
      <section className="w-full mt-6 py-6 px-[29px] bg-light dark:bg-[#393939]">
        <div>
          <p className="font-bold text-xl text-[#001A46] dark:text-light">
            مشخصات کاربری
          </p>
          <div className="flex items-center mt-4">
            <span className="w-[17%] h-0.5 bg-secondary-default dark:bg-primary-200"></span>
            <span className="w-[83%] h-0.5 bg-dark/[7%] dark:bg-[#4A4A4A]"></span>
          </div>
        </div>
        <div className="grid grid-cols-1 md:grid-cols-2 xl:grid-cols-3 gap-x-7 xl:gap-x-10 gap-y-7 xl:gap-y-[32px] mt-10">
          {inputs.map((input) => (
            <div
              key={input.id}
              className="child:text-[#696464] dark:child:text-light"
            >
              <p className="font text-lg">
                {input.label}{" "}
                {input.required ? (
                  <span className="text-error-400">*</span>
                ) : (
                  ""
                )}
              </p>
              <input
                className="w-full h-[53px] mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 p-4 text-base-mines rounded-sm"
                type={input.inputType}
                placeholder="وارد کنید"
                {...register(input.value)}
              />
              {errors[input.value] && (
                <div className="flex items-center gap-2.5 w-full mt-2.5 child:text-[#FF0000] text-xs">
                  <svg className="size-5 shrink-0">
                    <use href="#danger"></use>
                  </svg>
                  <p className="text-wrap whitespace-normal">
                    {errors[input.value]?.message}
                  </p>
                </div>
              )}
            </div>
          ))}
          <div className="child:text-[#696464] dark:child:text-light">
            <p className="font text-lg">جنسیت :</p>
            <select
              className="w-full h-[53px] mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 p-4 pl-10 text-base-mines rounded-sm outline-none"
              {...register("gender")}
            >
              <option value="مرد" defaultChecked>
                مرد
              </option>
              <option value="زن"> زن </option>
            </select>
            {errors.gender && (
              <div className="flex items-center gap-2.5 w-full mt-2.5 child:text-[#FF0000] text-xs">
                <svg className="size-5 shrink-0">
                  <use href="#danger"></use>
                </svg>
                <p className="text-wrap whitespace-normal">
                  {errors.gender?.message}
                </p>
              </div>
            )}
          </div>
          <div className="child:text-[#696464] dark:child:text-light">
            <p className="font text-lg">تاریخ تولد :</p>
            <Controller
              control={control}
              name="datebirth"
              rules={{ required: true }}
              render={({ field: { onChange } }) => (
                <div className="w-full h-full">
                  <div className="flex items-center justify-between w-full h-[53px] mt-2.25 bg-[#EFEFEF] dark:bg-dark-500 p-4 pl-10 text-base-mines rounded-sm outline-none">
                    <DatePicker
                      value={dateValue}
                      format="YYYY/MM/DD"
                      style={{
                        background: "inherit",
                        width: "150px",
                        height: "53px",
                        padding: "16px",
                      }}
                      className="!w-full"
                      onChange={(date) => {
                        if (date?.isValid) {
                          setDateValue(date);
                          const gregorianDate = date
                            .convert(gregorian)
                            .format("YYYY-MM-DD");
                          onChange(gregorianDate);
                        } else {
                          setDateValue("");
                          onChange("");
                        }
                      }}
                      placeholder="وارد کنید"
                      inputClass="datePicker_style"
                      calendar={persian}
                      locale={persian_fa}
                      onOpen={() => {
                        const input =
                          document.querySelector(".datePicker_style");
                        if (input) {
                          (input as HTMLInputElement).readOnly = true;
                          (input as HTMLInputElement).inputMode = "none";
                        }
                      }}
                      onClose={() => {
                        const input =
                          document.querySelector(".datePicker_style");
                        if (input) {
                          (input as HTMLInputElement).readOnly = true;
                          (input as HTMLInputElement).inputMode = "none";
                        }
                      }}
                    />
                    <svg className="w-5 h-5 text-tertiary-40 shrink-0">
                      <use href="#date"></use>
                    </svg>
                  </div>
                  <span className="user-info--error">
                    {errors.datebirth && errors.datebirth.message}
                  </span>
                </div>
              )}
            />
          </div>
          <div className="w-full mt-auto">
            <Button
              className="hidden sm:inline-block flex-center w-full h-[53px] px-5 bg-secondary-default hover:bg-secondary-default/90 text-base dark:text-light rounded"
              type="submit"
              disabled={sendUserInfo_isPending}
            >
              ذخیره
            </Button>
          </div>
        </div>
      </section>
    </form>
  );
}
